# -*- coding: utf-8 -*-
import collections
import json
import os
import tarfile

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, ProcessRegistry, subprocess
from sandbox.common.errors import TaskFailure

from sandbox.projects.adfox.qa.resource_types import AdfoxShmShootValidator
from sandbox.projects.adfox.qa.tasks.AdfoxServerFunctionalShoot import AdfoxServerFunctionalShoot
from sandbox.projects.adfox.qa.utils import constants
from sandbox.projects.adfox.qa.utils.constants import YAV_ROBOT_QA_ADFOX_YT_TOKEN
from sandbox.projects.adfox.qa.utils.secrets import get_yav_secret

# TODO reuse constants from validator
OK = u'0'
TEST_MISS = u'1'
PROD_MISS = u'2'
DIFF = u'3'


class AdfoxServerFunctionalShootValidate(sdk2.Task):
    class Context(sdk2.Task.Context):
        is_valid = True

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Validation settings') as validation_settings:
            validator_executable = sdk2.parameters.LastReleasedResource(
                'Validator executable that performs MapReduce operations',
                required=True, resource_type=AdfoxShmShootValidator)
            event_log_validation_threshold = sdk2.parameters.Integer(
                'Event log validation threshold', required=True)
            fraud_event_log_validation_threshold = sdk2.parameters.Integer(
                'Event log validation threshold', required=True)
            external_requests_log_validation_threshold = sdk2.parameters.Integer(
                'Event log validation threshold', required=True)
            testenv_switch_trigger = sdk2.parameters.String(
                'testenv_switch_trigger', required=True)

        with sdk2.parameters.Group('Task to validate') as task:
            task_to_validate = sdk2.parameters.Task('Task to validate', task_type=AdfoxServerFunctionalShoot,
                                                    required=True)

    def check_validator_results(self, tag, validator_results_file_path, threshold):
        results = collections.defaultdict(int)
        results.update(json.load(open(validator_results_file_path, 'r')))
        total = results[OK] + results[TEST_MISS] + results[PROD_MISS] + results[DIFF]
        fail = results[TEST_MISS] + results[PROD_MISS] + results[DIFF]
        percentage = float(fail) / float(total) * 100
        report = [
            'Report for {0}'.format(tag),
            'ok: {0}'.format(results[OK]),
            'diff: {0}'.format(results[DIFF]),
            'test miss: {0}'.format(results[TEST_MISS]),
            'production miss: {0}'.format(results[PROD_MISS]),
            'percentage: {0:.2f}'.format(percentage),
            'threshold: {0}'.format(threshold),
            'is_valid: {0}'.format(percentage <= threshold),
            ''
        ]
        self.set_info('\n'.join(report), do_escape=False)
        if percentage > threshold:
            self.Context.is_valid = False

    def on_execute(self):

        current_path = str(self.path())
        validator_tar_path = str(sdk2.ResourceData(self.Parameters.validator_executable).path)
        with tarfile.open(validator_tar_path) as tar:
            tar.extractall(path=current_path)
        validator_path = os.path.join(current_path, 'shoot_validator')

        env = dict(os.environ, YT_TOKEN=get_yav_secret(YAV_ROBOT_QA_ADFOX_YT_TOKEN)['YT_TOKEN'])
        task_context = self.Parameters.task_to_validate.Context

        tag = 'event_log'
        with ProcessRegistry, ProcessLog(self, logger='event_log') as process_log:
            options = [
                '--proxy', constants.YT_PROXY,
                '--production-table', '{}/{}/{}'.format(constants.YT_PRODUCTION_EVENT_LOG_DIR, 'adfox',
                                                        self.Parameters.testenv_switch_trigger),
                '--test-table', task_context.result_event_log
            ]
            subprocess.check_call([validator_path, 'validate_{0}'.format(tag)] + options,
                                  stdout=process_log.stdout,
                                  stderr=process_log.stderr,
                                  env=env)
            self.check_validator_results(tag, os.path.join(current_path, 'validate_{0}'.format(tag)),
                                         self.Parameters.event_log_validation_threshold)

        tag = 'external_requests_log'
        with ProcessRegistry, ProcessLog(self, logger='external_requests_log') as process_log:
            options = [
                '--proxy', constants.YT_PROXY,
                '--production-table', '{}/{}'.format(constants.YT_PRODUCTION_EXT_REQ_LOG_DIR,
                                                     self.Parameters.testenv_switch_trigger),
                '--prepared-log-table', '{}/{}/{}'.format(constants.YT_PREPARED_LOG_DIR, 'adfox',
                                                          self.Parameters.testenv_switch_trigger),
                '--test-table', task_context.result_external_requests_log
            ]
            subprocess.check_call([validator_path, 'validate_{0}'.format(tag)] + options,
                                  stdout=process_log.stdout,
                                  stderr=process_log.stderr,
                                  env=env)
            self.check_validator_results(tag, os.path.join(current_path, 'validate_{0}'.format(tag)),
                                         self.Parameters.external_requests_log_validation_threshold)

        if not self.Context.is_valid:
            raise TaskFailure('Shoot did not pass validation test!')
