from sandbox import sdk2

from sandbox.projects.yabs.qa.ammo_module.requestlog.adapters.general.sandbox import \
    AmmoRequestlogModuleGeneralSandboxAdapter as AmmoRequestlogModuleAdapter
from sandbox.projects.yabs.qa.dolbilo_module.simple.adapters.sandbox import \
    DolbiloModuleSandboxAdapter as DolbiloModuleAdapter
from sandbox.projects.yabs.qa.dumper_module.adapters.sandbox import DumperModuleSandboxAdapter
from sandbox.projects.yabs.qa.perf_module.adapters.sandbox import PerfModuleSandboxAdapter
from sandbox.projects.common.yabs.performance_best_cmp.stats import hodges_lehmann_diff, hodges_lehmann_median

from sandbox.projects.adfox.qa.resource_types import AdfoxShmFlameGraphResource
from sandbox.projects.adfox.qa.modules.amacs_module.adapters.sandbox import AmacsSandboxAdapter
from sandbox.projects.adfox.qa.tasks.AdfoxServerBaseShoot import AdfoxServerBaseShoot, SHOOT_CONTAINER
from sandbox.projects.adfox.qa.tasks.AdfoxServerPerformanceShoot import AdfoxServerPerformanceShoot
from sandbox.projects.adfox.qa.utils.shoot_utils import NoContext


# Actual performance shoot happens here for both engines
# Shoot is executed on one hardware for better results precision


class AdfoxServerPerformanceClosedShootCompare(sdk2.Task):
    name = 'ADFOX_SERVER_PERFORMANCE_CLOSED_SHOOT_COMPARE'

    class Context(sdk2.Context):
        testenv_database = None
        has_diff = False
        diff_percent = 0

    class Requirements(AdfoxServerBaseShoot.Requirements):
        pass

    class Parameters(sdk2.Parameters):
        _container = sdk2.parameters.Container(
            'lxc container for adfox shoot tasks',
            default_value=SHOOT_CONTAINER,
            platform='linux_ubuntu_16.04_xenial',
            required=True,
        )

        with sdk2.parameters.Group('Tasks to compare') as tasks:
            first_task = sdk2.parameters.Task('First task', task_type=AdfoxServerPerformanceShoot, required=True)
            second_task = sdk2.parameters.Task('Second task', task_type=AdfoxServerPerformanceShoot, required=True)

        with sdk2.parameters.Group('General settings') as general:
            dumper_module = DumperModuleSandboxAdapter.get_init_parameters_class()()
            shoot_sessions = sdk2.parameters.Integer('Number of shoots for each engine',
                                                     default_value=5, required=True)
            run_perf = sdk2.parameters.Bool('Run perf', default_value=False)
            with run_perf.value[True]:
                perf_module_parameters = PerfModuleSandboxAdapter.get_init_parameters_class()()
            diff_threshold = sdk2.parameters.Float('Diff threshold', default_value=0.02)

    def run_pipeline(self, parameters):
        """
        executes multiple "run_shoot" and processes returned reports
        :param parameters:
        :return:
        """
        # once for pipeline
        dumper_module = DumperModuleSandboxAdapter(self.Parameters.dumper_module, self).create_module()
        perf_module = PerfModuleSandboxAdapter(self.Parameters.perf_module_parameters, self).create_module()
        ammo_module = AmmoRequestlogModuleAdapter(parameters.ammo_parameters, self).create_module()
        shoot_module = DolbiloModuleAdapter(parameters.dolbilo_parameters, self).create_module()

        run_perf = self.Parameters.run_perf
        results = []
        for _ in range(self.Parameters.shoot_sessions):
            # for every session
            engine_service = AmacsSandboxAdapter(parameters.engine_parameters, self).create_module()
            with engine_service as active_service:
                pid = active_service.get_process().pid
                with (perf_module.collect_perf_data(pid) if run_perf else NoContext()) as perf_record_path:
                    dump_path = shoot_module.shoot(active_service,
                                                   ammo_module.get_dplan_path(),
                                                   store_dump=False)
                    results.append({
                        'processed_stats': dumper_module.get_processed_stats(dump_path),
                        'perf_record_path': perf_record_path
                    })

        if run_perf:
            best_result = max(results, key=lambda result: float(result['processed_stats']['rps']))
            perf_results_path = perf_module.generate_perf_results(best_result['perf_record_path'])
            resource = AdfoxShmFlameGraphResource(self, 'FlameGraph resource', perf_results_path)
            sdk2.ResourceData(resource).ready()

        rps_list = [float(r['processed_stats']['rps']) for r in results]
        return rps_list

    def to_context(self, suffix, result):
        for key, value in result.iteritems():
            setattr(self.Context, '{0}{1}'.format(key, suffix), value)

    def on_create(self):
        self.Parameters._container = self.Parameters.second_task.Parameters._container

    def on_execute(self):
        rps_list_1 = self.run_pipeline(self.Parameters.first_task.Parameters)
        rps_list_2 = self.run_pipeline(self.Parameters.second_task.Parameters)
        rps_hl_1, rps_hl_2 = hodges_lehmann_median(rps_list_1), hodges_lehmann_median(rps_list_2)
        rps_hl_diff = hodges_lehmann_diff(rps_list_1, rps_list_2)
        diff_percent = 100 * rps_hl_diff / rps_hl_1
        self.Context.diff_percent = diff_percent
        report_lines = ["Max requests/sec changed by {:+.4f} % ({} -> {}), threshold: {:+.4f} %".format(
            diff_percent, rps_hl_1, rps_hl_2, self.Parameters.diff_threshold * 100
        )]
        if diff_percent > self.Parameters.diff_threshold * 100:
            self.Context.has_diff = True
            report_lines.append('DIFF! diff_percent: {0} > diff_threshold: {1}'.format(
                diff_percent, self.Parameters.diff_threshold * 100))
        self.set_info('\n'.join(report_lines))
