from sandbox import sdk2

from sandbox.projects.yabs.qa.ammo_module.requestlog.adapters.general.sandbox import \
    AmmoRequestlogModuleGeneralSandboxAdapter as AmmoRequestlogModuleAdapter
from sandbox.projects.yabs.qa.dolbilo_module.simple.adapters.sandbox import \
    DolbiloModuleSandboxAdapter as DolbiloModuleAdapter
from sandbox.projects.yabs.qa.dumper_module.adapters.sandbox import DumperModuleSandboxAdapter
from sandbox.projects.yabs.qa.perf_module.adapters.sandbox import PerfModuleSandboxAdapter

from sandbox.projects.adfox.qa.modules.amacs_module.adapters.sandbox import AmacsSandboxAdapter
from sandbox.projects.adfox.qa.tasks.AdfoxServerBaseShoot import AdfoxServerBaseShoot, SHOOT_CONTAINER
from sandbox.projects.adfox.qa.tasks.AdfoxServerPerformanceShoot import AdfoxServerPerformanceShoot
from sandbox.projects.adfox.qa.utils.shoot_utils import NoContext

# Actual performance shoot happens here for both engines
# Shoot is executed on one hardware for better results precision

COMPARE_FIELD = 'user_work_time'


class AdfoxServerPerformanceOpenShootCompare(sdk2.Task):
    name = 'ADFOX_SERVER_PERFORMANCE_OPEN_SHOOT_COMPARE'

    class Context(sdk2.Context):
        testenv_database = None
        has_diff = False

    class Requirements(AdfoxServerBaseShoot.Requirements):
        pass

    class Parameters(sdk2.Parameters):
        _container = sdk2.parameters.Container(
            'lxc container for adfox shoot tasks',
            default_value=SHOOT_CONTAINER,
            platform='linux_ubuntu_16.04_xenial',
            required=True,
        )

        with sdk2.parameters.Group('Tasks to compare') as tasks:
            first_task = sdk2.parameters.Task('First task', task_type=AdfoxServerPerformanceShoot, required=True)
            second_task = sdk2.parameters.Task('Second task', task_type=AdfoxServerPerformanceShoot, required=True)

        with sdk2.parameters.Group('General settings') as general:
            dumper_module = DumperModuleSandboxAdapter.get_init_parameters_class()()
            rps = sdk2.parameters.Integer('RPS', default_value=1000, required=True)
            shoot_time = sdk2.parameters.Integer('Shoot time in seconds',
                                                 default_value=60 * 60, required=True)
            run_perf = sdk2.parameters.Bool('Run perf', default_value=False)
            with run_perf.value[True]:
                perf_module_parameters = PerfModuleSandboxAdapter.get_init_parameters_class()()
            diff_threshold = sdk2.parameters.Float('Diff threshold', default_value=5.0)

    def measure_engine_starting_time(self, engine):
        with engine:
            pass
        return engine.work_time, engine.user_work_time, engine.system_work_time

    def run_pipeline(self, parameters):
        # once for pipeline
        dumper_module = DumperModuleSandboxAdapter(self.Parameters.dumper_module, self).create_module()
        perf_module = PerfModuleSandboxAdapter(self.Parameters.perf_module_parameters, self).create_module()
        ammo_module = AmmoRequestlogModuleAdapter(parameters.ammo_parameters, self).create_module()

        shoot_module = DolbiloModuleAdapter(parameters.dolbilo_parameters, self).create_module()

        # setup module for shootingwith fixed rps
        shoot_module.adapter.parameters.circular_session = True  # reuse bullets
        shoot_module.adapter.parameters.shoot_request_limit = 0  # disable fire limit
        shoot_module.adapter.parameters.mode = 'plan'  # shoot mode
        # fixed rps and shooting time
        shoot_module.adapter.parameters.mode_arg = ['--rps-schedule', 'const({}, {})'.format(self.Parameters.rps,
                                                                                             self.Parameters.shoot_time)]

        run_perf = self.Parameters.run_perf
        engine_service = AmacsSandboxAdapter(parameters.engine_parameters, self).create_module()
        engine_starting_time, engine_user_starting_time, engine_system_starting_time = self.measure_engine_starting_time(
            engine_service)
        with engine_service as active_service:
            pid = active_service.get_process().pid
            with (perf_module.collect_perf_data(pid) if run_perf else NoContext()) as perf_record_path:
                dump_path = shoot_module.shoot(active_service,
                                               ammo_module.get_dplan_path(),
                                               store_dump=False)
        result = {
            'processed_stats': dumper_module.get_processed_stats(dump_path),
            'perf_record_path': perf_record_path,
            'engine_starting_time': engine_starting_time,
            'engine_working_time': engine_service.work_time - engine_starting_time,
            'engine_avg_work_time': (engine_service.work_time - engine_starting_time) / float(
                dumper_module.get_processed_stats(dump_path)["requests"]),
            'system_work_time': engine_service.system_work_time - engine_system_starting_time,
            'user_work_time': engine_service.user_work_time - engine_user_starting_time,
            'rusage': engine_service.rusage,
        }
        if run_perf:
            # TODO: generate resource
            perf_module.generate_perf_results(result['perf_record_path'])
        return result

    def to_context(self, suffix, result):
        for key, value in result.iteritems():
            setattr(self.Context, '{0}{1}'.format(key, suffix), str(value))

    def create_procent_diff(self, fields, result_1, result_2):
        result_diff = None
        for field in fields:
            test_result_diff = abs(result_1[field] - result_2[field])
            working_time_percent_diff = test_result_diff / max(result_1[field], result_2[field]) * 100
            if field == COMPARE_FIELD:
                result_diff = working_time_percent_diff
            setattr(self.Context, field + '_difference_in_percent', str(working_time_percent_diff) + '%')
        return result_diff

    def on_create(self):
        self.Parameters._container = self.Parameters.second_task.Parameters._container

    def on_execute(self):
        result_1 = self.run_pipeline(self.Parameters.first_task.Parameters)
        result_2 = self.run_pipeline(self.Parameters.second_task.Parameters)
        self.to_context(suffix='_1', result=result_1)
        self.to_context(suffix='_2', result=result_2)
        diff_percentage = self.create_procent_diff(['engine_working_time', 'system_work_time', 'user_work_time'],
                                                   result_1, result_2)
        if diff_percentage > self.Parameters.diff_threshold:
            self.Context.has_diff = True
        self.set_info('Threshold: {0}, difference: {1}, has_diff: {2}'.format(self.Parameters.diff_threshold,
                                                                              diff_percentage, self.Context.has_diff))
        self.set_info('Shoot time for current commit: {}'.format(result_1[COMPARE_FIELD]))
        self.set_info('Shoot time for new commit: {}'.format(result_2[COMPARE_FIELD]))
