# -*- coding: utf-8 -*-

import logging
import os
import shutil

from sandbox.projects import resource_types

from sandbox import sdk2
from sandbox.projects.adfox.resource_types import AdfoxEngineConfigurationLayer
from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess, ProcessLog
from sandbox.sdk2.vcs import svn

CONTAINER_WITH_MYSQL = 829148115


class AdfoxServerPrepareConfiguration(sdk2.Task):
    name = "ADFOX_SERVER_PREPARE_CONFIGURATION"

    class Requirements(sdk2.Task.Requirements):
        cores = 4
        disk_space = 10 * 1024
        ram = 4 * 1024
        privileged = True  # for container
        environments = (
            environments.PipEnvironment("pymysql", "0.9.3"),
        )

    class Parameters(sdk2.Task.Parameters):
        _container = sdk2.parameters.Container("LXC Container with adfox-db-5.7", default_value=CONTAINER_WITH_MYSQL,
                                               platform="linux_ubuntu_16.04_xenial", required=True)

        checkout_arcadia_from_url = sdk2.parameters.String("Get Arcadia via URL", required=False,
                                                           default_value="arcadia:/arc/trunk/arcadia")
        arcadia_patch = sdk2.parameters.String("Arcadia patch URL", required=False)

    class Context(sdk2.Task.Context):
        arc_repo_path = None
        migrator_build_task = None
        controller_build_task = None

    @property
    def git_repo_path(self):
        return os.path.join(str(self.path()), "db")

    def _prepare_container(self):
        with ProcessLog(self, "prepare_container") as log:
            subprocess.check_call(["timedatectl", "set-timezone", "Europe/Moscow"],
                                  stdout=log.stdout, stderr=log.stderr)
            subprocess.check_call(["/opt/adfox-sql/bin/virgin_mysql.sh"], stdout=log.stdout, stderr=log.stderr)
            subprocess.check_call(["git", "clone", "http://git.adfox.ru/adfox/db.git", self.git_repo_path],
                                  stdout=log.stdout, stderr=log.stderr)

        # TODO: do we need it? leave for basic version
        import pymysql

        connection = pymysql.connect(host="localhost", port=3306, user="adfox",
                                     password="", charset="cp1251")
        with connection.cursor() as cursor:
            cursor.execute("SET GLOBAL sql_mode = 'NO_ENGINE_SUBSTITUTION';")
        connection.commit()
        connection.close()

    def _run_migrations(self):
        with ProcessLog(self, "migrations") as log:
            subprocess.check_call(["php", "phinx", "migrate", "-c", "migrations/sandbox_mainDB.yml"],
                                  cwd=self.git_repo_path, stdout=log.stdout, stderr=log.stderr)
            subprocess.check_call(["php", "phinx", "migrate", "-c", "migrations/sandbox_commonDB.yml"],
                                  cwd=self.git_repo_path, stdout=log.stdout, stderr=log.stderr)

    def on_save(self):
        splitted = str(self.Parameters.checkout_arcadia_from_url).split('@')
        if len(splitted) > 2:
            revision = int(splitted[-1])
            if revision < 6838991:
                self.Parameters.checkout_arcadia_from_url = str(
                    self.Parameters.checkout_arcadia_from_url).replace(splitted[-1], "6850874")

    def on_prepare(self):
        # build resources
        with self.memoize_stage.launch_build_tasks:
            self.Context.migrator_build_task = \
                self.launch_build_task("adfox/infra/amacs_config/migrator", "config_migrator")
            self.Context.controller_build_task = \
                self.launch_build_task("adfox/infra/amacs_config/cfg_ctl", "cfg_ctl")
        check_tasks(self, tasks=[self.Context.migrator_build_task,
                                 self.Context.controller_build_task])

        # prepare arcadia
        self.Context.arc_repo_path = svn.Arcadia.get_arcadia_src_dir(
            "{0}".format(self.Parameters.checkout_arcadia_from_url))

        # apply patch
        if self.Parameters.arcadia_patch:
            svn.Arcadia.apply_patch(
                self.Context.arc_repo_path,
                str(self.Parameters.arcadia_patch),
                str(self.path())
            )

    def launch_build_task(self, path_in_arcadia, build_artifact):
        build_task = sdk2.Task["YA_MAKE_2"](
            self, owner=self.owner, description="build {0}".format(path_in_arcadia),
            checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
            use_aapi_fuse=True, aapi_fallback=True,
            checkout_mode="auto", targets=path_in_arcadia,
            arts=os.path.join(path_in_arcadia, build_artifact)
        ).save().enqueue()
        return build_task.id

    def on_execute(self):
        # get resources
        migrator_resource = sdk2.Resource.find(
            resource_type=resource_types.ARCADIA_PROJECT,
            task_id=self.Context.migrator_build_task
        ).first()

        controller_resource = sdk2.Resource.find(
            resource_type=resource_types.ARCADIA_PROJECT,
            task_id=self.Context.controller_build_task
        ).first()

        migrator_path = os.path.join(str(sdk2.ResourceData(migrator_resource).path), "config_migrator")
        controller_path = os.path.join(str(sdk2.ResourceData(controller_resource).path), "cfg_ctl")
        logging.info(migrator_path)

        # prepare container
        self._prepare_container()
        self._run_migrations()

        # clone migrations files
        arc_repo_path = os.path.join(str(self.path()), "arc_repo")
        shutil.copytree(os.path.join(self.Context.arc_repo_path, "adfox/infra/amacs_config"), arc_repo_path)

        # run configuration migrations
        with ProcessLog(self, "migrator") as log:
            # launch migrator in "correct" directory
            subprocess.check_call(
                [migrator_path, "--db-host", "localhost", "--db-user", "root",
                 "--sandbox", "migrate", "--yes"],
                cwd=os.path.join(arc_repo_path, "migrator"),
                stdout=log.stdout, stderr=log.stderr
            )

        # prepare configuration file
        configuration_dir = str(self.path())
        configuration_file = os.path.join(configuration_dir, "engine.lua")
        with ProcessLog(self, "controller") as log:
            # launch controller in "correct" directory
            subprocess.check_call(
                [controller_path, "--env", "shm", "--config_dir", configuration_dir,
                 "--db-host", "localhost", "--db-user", "root", "--sandbox"],
                cwd=os.path.join(arc_repo_path, "cfg_ctl"),
                stdout=log.stdout, stderr=log.stderr
            )

        # create configuration resource
        sdk2.ResourceData(AdfoxEngineConfigurationLayer(
            self, "engine configuration file", configuration_file
        )).ready()
