import datetime
import json
import logging
import os
import tarfile

from sandbox import sdk2
from sandbox.projects.adfox.qa.resource_types import AdfoxBulletsBuilder, AdfoxShmAdfoxSpec
from sandbox.projects.adfox.qa.tasks.AdfoxServerBase import AdfoxServerBase
from sandbox.projects.adfox.qa.tasks.AdfoxServerShmCreator import query
from sandbox.projects.adfox.qa.utils.constants import SB_ROBOT_QA_ADFOX_YAV_OAUTH_TOKEN
from sandbox.projects.adfox.qa.utils.secrets import get_sb_secret
from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.sandbox.resources import LXC_CONTAINER
from sandbox.projects.yabs.qa.tasks.BuildAmmoAndStubsFromYT import BuildAmmoAndStubsFromYT
from sandbox.projects.yql.RunYQL2 import RunYQL2
from sandbox.sdk2.helpers import ProcessLog, subprocess


def lf_table(prefix, table_date):
    if datetime.datetime.now() - table_date < datetime.timedelta(days=2):
        table = '1h/{0}'.format(table_date.strftime('%Y-%m-%dT%H:00:00'))
    else:
        table = '1d/{0}'.format(table_date.strftime('%Y-%m-%d'))
    return '{0}/{1}'.format(prefix, table)


def yql_table(cluster, table):
    return '{0}.`{1}`'.format(cluster, table)


class AdfoxServerShmCreator(AdfoxServerBase):
    name = 'ADFOX_SERVER_SHM_CREATOR'

    class Requirements(AdfoxServerBase.Requirements):
        pass

    class Parameters(AdfoxServerBase.Parameters):
        testenv_switch_trigger = sdk2.parameters.String('testenv_switch_trigger', required=True)
        ext_req_table = sdk2.parameters.String('ext_req_table table path', required=True)
        yt_cluster = sdk2.parameters.String('YT cluster', required=True)

        cache_resource_id = sdk2.parameters.Integer('atlas_cache resource id', required=True)
        blacklists_resource_id = sdk2.parameters.Integer('blacklists resource id', required=True)
        engine_layer_resource_id = sdk2.parameters.Integer('engine layer resource id', required=True)
        configuration_layer_resource_id = sdk2.parameters.Integer('configuration layer resource id', required=True)
        atlas_to_database_resource_id = sdk2.parameters.Integer('atlas_to_database resource id', required=True)

        bullets_builder_revision = sdk2.parameters.Integer('bullets_builder revision', required=True)

        reuse_collected_logs = sdk2.parameters.Bool('reuse collected_logs', required=True, default=True)

    class Context(AdfoxServerBase.Context):
        # contains information about launched tasks
        collected_logs_task = None
        bullets_builder_task = None
        create_resources_task = None
        filter_event_log_task = None
        create_container_task = None

    def on_execute(self):

        yt_cluster = self.Parameters.yt_cluster
        ext_req_table = self.Parameters.ext_req_table
        collected_logs_table = ext_req_table.replace('external_requests_logs', 'collected_logs')

        # build collected_logs from access_log and external_requests_log
        # collected_logs can be reused for different spec types
        with self.memoize_stage.build_collected_logs(commit_on_entrance=False, commit_on_wait=False):

            if not self.yt.exists(collected_logs_table) or not self.Parameters.reuse_collected_logs:
                # wait access_log
                table_date = datetime.datetime.strptime(str(self.Parameters.testenv_switch_trigger),
                                                        '%y-%m-%dT%H:%M:%S')
                access_logs_table = lf_table('//home/logfeller/logs/adfox-nginx-access-log', table_date)
                if not self.yt.exists(access_logs_table):
                    logging.info('wait access log')
                    raise sdk2.WaitTime(datetime.timedelta(minutes=20).total_seconds())

                event_log_table = lf_table('//home/logfeller/logs/adfox-event-log', table_date)
                if not self.yt.exists(event_log_table):
                    logging.info('wait event log')
                    raise sdk2.WaitTime(datetime.timedelta(minutes=20).total_seconds())

                # create collected_logs
                filled_query = query.COLLECTED_LOGS_TEMPLATE \
                    .replace('#EXTERNAL_REQUESTS_LOG#', yql_table(yt_cluster, ext_req_table)) \
                    .replace('#ACCESS_LOG#', yql_table(yt_cluster, access_logs_table)) \
                    .replace('#EVENT_LOG#', yql_table(yt_cluster, event_log_table)) \
                    .replace('#COLLECTED_LOG#', yql_table(yt_cluster, collected_logs_table))
                collected_logs_task = RunYQL2(self, description='build collected_logs', owner='ADFOX',
                                              use_v1_syntax=True, trace_query=True, publish_query=True,
                                              query=filled_query).save().enqueue()
                self.Context.collected_logs_task = collected_logs_task.id
                self.set_info('creating collected_logs in {0}'.format(self.Context.collected_logs_task))

            else:
                self.set_info('using existing collected_logs')
        if self.Context.collected_logs_task:
            check_tasks(self, tasks=[self.Context.collected_logs_task])

        # build prepared_logs from collected logs using ADFOX_BULLETS_BUILDER
        # prepared_logs can't be reused for different spec types - ADFOX_BULLETS_BUILDER produces
        # different results for different spec types
        with self.memoize_stage.build_prepared_logs(commit_on_entrance=False, commit_on_wait=False):
            # build resource
            with self.memoize_stage.build_bullet_builder(commit_on_entrance=False):
                build_task = sdk2.Task["YA_PACKAGE"](
                    self,
                    owner=self.owner,
                    description='build bullet_builder',
                    checkout_arcadia_from_url='arcadia:/arc/trunk/arcadia',
                    packages='adfox/qa/shm/creator/bullets_builder/package.json',
                    package_type='tarball',
                    resource_type='ADFOX_BULLETS_BUILDER'
                ).save().enqueue()
                self.Context.bullets_builder_task = build_task.id
                self.set_info('building bullets_builder in {0}'.format(self.Context.bullets_builder_task))
            check_tasks(self, tasks=[self.Context.bullets_builder_task])

            # get resource
            bullets_builder_resource = sdk2.Resource.find(
                resource_type=AdfoxBulletsBuilder,
                task_id=self.Context.bullets_builder_task
            ).first()
            bullets_builder_path = str(sdk2.ResourceData(bullets_builder_resource).path)
            with tarfile.open(bullets_builder_path) as tar:
                tar.extractall(path=str(self.path()))
            bullets_builder_executable = os.path.join(str(self.path()), 'bullets_builder')

            # running bullet_builder
            self.set_info('running bullets_builder')
            yav_oauth_token = get_sb_secret(SB_ROBOT_QA_ADFOX_YAV_OAUTH_TOKEN)

            def set_required_attributes(table_path):
                """Sets specific attributes for BUILD_AMMO_AND_STUBS_FROM_YT task"""
                self.yt.set_attribute(table_path, 'cachedaemon_key_headers',
                                      ["X-Adfox-Request-ID", "X-Adfox-Request-Index",
                                       "X-Adfox-Request-Session"])
                self.yt.set_attribute(table_path, 'request_key_headers',
                                      ["X-Adfox-Request-ID", "X-Adfox-Request-Time",
                                       "X-Adfox-Debug", "X-Adfox-Random-Seed"])

            def run_bullet_builder():
                prepared_logs_table = ext_req_table.replace('external_requests_logs',
                                                            'prepared_logs/adfox')
                with ProcessLog(logger='bullet_builder_log') as log:
                    args = [
                        bullets_builder_executable,
                        'remote',
                        '--source-table', collected_logs_table,
                        '--destination-table', prepared_logs_table
                    ]
                    subprocess.check_call(args, stdout=log.stdout, stderr=log.stderr,
                                          env={'YAV_OAUTH_TOKEN': yav_oauth_token})
                set_required_attributes(prepared_logs_table)

            run_bullet_builder()

        # use YABS task BUILD_AMMO_AND_STUBS_FROM_YT to generate requestlog and cachedaemon stub
        # from prepared_log tables
        with self.memoize_stage.build_bullets(commit_on_entrance=False):
            self.set_info('creating bullets and stubs')

            def run_build_bullets():
                prepared_logs_table = ext_req_table.replace('external_requests_logs',
                                                            'prepared_logs/adfox')
                task_id = BuildAmmoAndStubsFromYT(
                    self,
                    owner=self.owner,
                    description='create bullets and stubs resources',
                    cache_daemon_resource=845513904,
                    input_tables={"adfox": [prepared_logs_table]},
                    spec={"adfox": {"make_load": False}},
                    additional_attributes={"ttl": "inf"},
                    yt_token_vault_name='yabs-cs-sb-yt-token',
                    testenv_switch_trigger=False,
                    legacy_mode=False
                ).save().enqueue().id
                self.set_info('started BUILD_AMMO_AND_STUBS_FROM_YT #{0}'.format(task_id))
                self.Context.create_resources_task = task_id

            run_build_bullets()

        check_tasks(self, [self.Context.create_resources_task])

        with self.memoize_stage.filter_event_log(commit_on_entrance=False, commit_on_wait=False):

            table_date = datetime.datetime.strptime(str(self.Parameters.testenv_switch_trigger), '%y-%m-%dT%H:%M:%S')
            event_logs_table = lf_table('//home/logfeller/logs/adfox-event-log', table_date)
            if not self.yt.exists(event_logs_table):
                logging.info('wait event log')
                raise sdk2.WaitTime(datetime.timedelta(minutes=20).total_seconds())

            def run_filter_event_log():
                prepared_logs_table = ext_req_table.replace('external_requests_logs',
                                                            'prepared_logs/adfox')
                filtered_event_logs_table = ext_req_table.replace('external_requests_logs',
                                                                  'event_logs/adfox')
                filled_query = query.FILTER_EVENT_LOG_TEMPLATE \
                    .replace('#PREPARED_LOGS#', yql_table(yt_cluster, prepared_logs_table)) \
                    .replace('#EVENT_LOG#', yql_table(yt_cluster, event_logs_table)) \
                    .replace('#FILTERED_EVENT_LOG#', yql_table(yt_cluster, filtered_event_logs_table))
                filter_event_log_task = RunYQL2(self, description='filter event-log', owner='ADFOX',
                                                use_v1_syntax=True, trace_query=True, publish_query=True,
                                                query=filled_query).save().enqueue()
                self.Context.filter_event_log_task = filter_event_log_task.id

            self.set_info('filter adfox-event-log for validation tasks')

            run_filter_event_log()

        check_tasks(self, [self.Context.filter_event_log_task])

        # create container resource
        with self.memoize_stage.create_conteiner_resource(commit_on_entrance=False, commit_on_wait=False):
            dir_path = os.path.dirname(os.path.realpath(__file__))
            with open(os.path.join(dir_path, 'container.sh'), 'r') as container_script_file:
                container_script = container_script_file.read()
            container_script = container_script.replace('#blacklists_resource_id#',
                                                        str(self.Parameters.blacklists_resource_id))
            task = sdk2.Task["SANDBOX_LXC_IMAGE"](
                self, owner=self.owner,
                description='create lxc container for shoot tasks',
                resource_type="LXC_CONTAINER",
                resource_description="LXC container for ADFOX shoot tests",
                ubuntu_release="xenial", test_result_lxc=True, set_default=False,
                push_to_preprod=False, custom_image=True, install_common=False,
                custom_script=container_script
            ).save().enqueue()
            self.Context.create_container_task = task.id
            self.set_info('creating lxc container for shoot tasks in {0}'.format(self.Context.create_container_task))
        check_tasks(self, [self.Context.create_container_task])
        container_resource = sdk2.Resource.find(
            resource_type=LXC_CONTAINER,
            task_id=self.Context.create_container_task
        ).first()

        # generate spec file
        with self.memoize_stage.generate_spec(commit_on_entrance=False, commit_on_wait=False):
            self.set_info('generating spec')

            def generate_spec_file(spec_cls, spec_data, resource_attributes):
                spec_file_path = os.path.join(str(self.path()), 'spec')
                with open(spec_file_path, 'w') as spec_file:
                    json.dump(spec_data, spec_file)
                sdk2.ResourceData(spec_cls(
                    self, 'adfox server spec', spec_file_path, **resource_attributes
                )).ready()

            build_task = self.find(id=self.Context.create_resources_task).first()
            generate_spec_file(spec_cls=AdfoxShmAdfoxSpec, spec_data={
                "atlas_resource_id": self.Parameters.cache_resource_id,
                "atlas_to_database_resource_id": self.Parameters.atlas_to_database_resource_id,
                "responses_resource_id": build_task.Parameters.result_spec["adfox"]["stub_resource"],
                "requests_resource_id": build_task.Parameters.result_spec["adfox"]["ammo_resource"],
                "testenv_switch_trigger": self.Parameters.testenv_switch_trigger,
                "container_resource_id": container_resource.id,
            }, resource_attributes={
                "testenv_switch_trigger_adfox": self.Parameters.testenv_switch_trigger
            })
