# -*- coding: utf-8 -*-
import datetime
import shutil
import os
import tarfile

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import ProcessRegistry, subprocess, ProcessLog
from sandbox.common.types.client import Tag
import sandbox.common.types.resource as ctr

from sandbox.projects.adfox.qa.tasks.AdfoxServerUpdateAtlasStructure import queries
from sandbox.projects.adfox.resource_types import AdfoxEngineBinaryLayer, AdfoxAtlasCacheFile, AdfoxAtlasToDbUploader


class AdfoxServerUpdateAtlasStructure(sdk2.Task):
    name = 'ADFOX_SERVER_UPDATE_ATLAS_STRUCTURE'

    class Requirements(sdk2.Task.Requirements):
        cores = 16
        disk_space = 100 * 1024  # 100 GB
        ram = 16 * 1024  # 16 GB
        privileged = True  # for container
        client_tags = Tag.SSD
        environments = (
            environments.PipEnvironment('pymysql', '0.9.3'),
        )

    class Parameters(sdk2.Task.Parameters):

        # Using data from adfox-amacs-mysql=2.0.20181228.140654 (installed in container)
        _container = sdk2.parameters.Container('LXC Container with adfox-db-5.7', default_value=829148115,
                                               platform='linux_ubuntu_16.04_xenial', required=True)
        adfox_server = sdk2.parameters.Resource('Resource with amacs binary ', resource_type=AdfoxEngineBinaryLayer,
                                                required=True)
        atlas_to_database = sdk2.parameters.Resource('Resource with atlas_to_database',
                                                     resource_type=AdfoxAtlasToDbUploader,
                                                     required=True)
        atlas_cache_file = sdk2.parameters.Resource('Production atlas cache file', resource_type=AdfoxAtlasCacheFile,
                                                    required=True)
        reuse_existing = sdk2.parameters.Bool('Reuse existing resources', default_value=True)

        with sdk2.parameters.Output:
            updated_atlas_cache_file = sdk2.parameters.Resource('Atlas cache file with updated structure',
                                                                resource_type=AdfoxAtlasCacheFile,
                                                                required=True)

    class Context(sdk2.Task.Context):
        atlas_sources_hash = None
        creation_timestamp = None

    @property
    def engine_path(self):
        return os.path.join(str(self.path()), 'amacs')

    @property
    def repo_path(self):
        return os.path.join(str(self.path()), 'db')

    @property
    def resource_path(self):
        return os.path.join(str(self.path()), 'created_atlas_file')

    @property
    def atlas_to_database_path(self):
        return os.path.join(str(self.path()), 'atlas_to_database')

    def _prepare_engine(self):
        # unpack engine first

        work_dir = str(self.path())
        engine_tar_path = str(sdk2.ResourceData(self.Parameters.adfox_server).path)
        with tarfile.open(engine_tar_path) as tar:
            tar.extractall(path=work_dir)
        if os.path.exists(os.path.join(work_dir, "adfox/engine/amacs")):
            shutil.move(os.path.join(work_dir, "adfox/engine/amacs"), work_dir)
            shutil.move(os.path.join(work_dir, "adfox/engine/lua"), work_dir)
        if not os.path.exists(self.engine_path) or not os.path.isfile(self.engine_path):
            raise Exception('Engine executable was not provided by resource!')

    def _get_atlas_sources_hash(self):
        # get atlas hash
        with ProcessRegistry:
            out = subprocess.check_output([self.engine_path, '--slave', '-v'])
            self.Context.atlas_sources_hash = out.splitlines()[-1].split(':')[-1].strip()

    def _check_existing_resources(self):
        # check if resource with revision = self.Parameters.adfox_server.svn_revision
        # and creation_timestamp = self.Parameters.atlas_cache_file.creation_timestamp
        # was already created and we don't need to create it again

        required_attrs = {'creation_timestamp': self.Parameters.atlas_cache_file.creation_timestamp,
                          'sources_hash': self.Context.atlas_sources_hash,
                          'target': 'production.updated'}  # at least on run happens
        existing_resource = sdk2.Resource.find(
            type=AdfoxAtlasCacheFile,
            state=ctr.State.READY,
            attrs=required_attrs
        ).order(-sdk2.Resource.id).first()  # ensure that latest is taken
        if existing_resource:
            self.Parameters.updated_atlas_cache_file = existing_resource
            return True
        return False

    def _prepare_container(self):
        with ProcessLog(self, 'prepare_container') as log:
            subprocess.check_call(['timedatectl', 'set-timezone', 'Europe/Moscow'],
                                  stdout=log.stdout, stderr=log.stderr)
            subprocess.check_call(['/opt/adfox-sql/bin/virgin_mysql.sh'], stdout=log.stdout, stderr=log.stderr)
            subprocess.check_call(['git', 'clone', 'http://git.adfox.ru/adfox/db.git', self.repo_path],
                                  stdout=log.stdout, stderr=log.stderr)
        queries.disable_database_strict_mode()

    def _run_migrations(self, date):
        with ProcessLog(self, 'old_migrations') as log:
            version = queries.get_migration_version(migration_date=date, database='adfox')
            subprocess.check_call(['php', 'phinx', 'migrate', '-c', 'migrations/sandbox_mainDB.yml', '-t', version],
                                  cwd=self.repo_path, stdout=log.stdout, stderr=log.stderr)
            version = queries.get_migration_version(migration_date=date, database='common')
            subprocess.check_call(['php', 'phinx', 'migrate', '-c', 'migrations/sandbox_commonDB.yml', '-t', version],
                                  cwd=self.repo_path, stdout=log.stdout, stderr=log.stderr)
            subprocess.check_call(r"mysql --skip-column-names adfox -e 'SHOW TRIGGERS;' | cut -f1 | "
                                  r"sed -r 's/(.*)/DROP TRIGGER IF EXISTS \1;/' | mysql adfox",
                                  shell=True, stdout=log.stdout, stderr=log.stderr)
            subprocess.check_call(r"mysql --skip-column-names common -e 'SHOW TRIGGERS;' | cut -f1 | "
                                  r"sed -r 's/(.*)/DROP TRIGGER IF EXISTS \1;/' | mysql common",
                                  shell=True, stdout=log.stdout, stderr=log.stderr)

    def _run_atlas_to_database(self):
        with tarfile.open(str(sdk2.ResourceData(self.Parameters.atlas_to_database).path)) as tar:
            tar.extractall(path=str(self.path()))
        with ProcessLog(self, 'atlas_to_database') as log:
            subprocess.check_call([self.atlas_to_database_path, 'upload',
                                   str(sdk2.ResourceData(self.Parameters.atlas_cache_file).path),
                                   '--log-level', 'debug'], stdout=log.stdout, stderr=log.stderr)

    def _create_atlas_cache(self):
        # don't update creation_timestamp for repeatable engine behaviour
        self.Context.creation_timestamp = self.Parameters.atlas_cache_file.creation_timestamp

        configurations_folder = os.path.join(str(self.path()), 'lua.d')
        os.makedirs(configurations_folder)
        required = [  # required, but not used configuration, inadequate values
            'bridge.config.host_id = 1',
            'bridge.config.engine_ids = {2, 3, 4}',
            'bridge.config.cache_addr = "localhost:80"',
            'bridge.config.counters_addr = "localhost:81"',
        ]
        configuration = required + [
            'bridge.config.atlas.initial_size = 6442450944',
            'bridge.config.atlas.max_size = 6442450944',
            'bridge.config.mysql_master.enable_keepalive = false',
            'bridge.config.mysql_master.database = "adfox"',
            'bridge.config.mysql_master.username = "adfox"',
            'bridge.config.mysql_master.address = "127.0.0.1:3306"',
            'bridge.config.force_creation_timestamp = {0}'.format(self.Context.creation_timestamp)
        ]
        with open(os.path.join(configurations_folder, 'cache.lua'), 'w') as cache_lua:
            cache_lua.write('\n'.join(configuration))
        with ProcessLog(self, 'bridge_save_atlas') as log:
            subprocess.check_call([self.engine_path, '--stderr', '--bridge', '--dump', self.resource_path],
                                  stdout=log.stdout, stderr=log.stderr)

    def on_execute(self):
        self._prepare_engine()
        self._get_atlas_sources_hash()

        if self.Parameters.reuse_existing and self._check_existing_resources():
            return

        # generating new atlas cache
        self._prepare_container()
        self._run_migrations(self.Parameters.atlas_cache_file.created)
        self._run_atlas_to_database()
        self._run_migrations(datetime.datetime.now())
        self._create_atlas_cache()

        # save resource
        created_resource = AdfoxAtlasCacheFile(self, 'atlas file', self.resource_path,
                                               revision=self.Parameters.adfox_server.svn_revision,
                                               target='production.updated',
                                               sources_hash=self.Context.atlas_sources_hash,
                                               creation_timestamp=self.Context.creation_timestamp)
        sdk2.ResourceData(created_resource).ready()
        self.Parameters.updated_atlas_cache_file = created_resource
