# -*- coding: utf-8 -*-
import os
import tarfile

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess, ProcessLog
from sandbox.common.types.client import Tag

from sandbox.projects.adfox.resource_types import AdfoxAtlasCacheFile, AdfoxAtlasToDbUploader


def ensure_path(path):
    if not os.path.exists(path):
        os.makedirs(path)
    return path


class AdfoxServerUpdateAtlasStructureCheck(sdk2.Task):
    name = 'ADFOX_SERVER_UPDATE_ATLAS_STRUCTURE_CHECK'

    class Requirements(sdk2.Task.Requirements):
        cores = 16
        disk_space = 8 * 1024  # 8 GB
        ram = 16 * 1024  # 16 GB
        privileged = True  # for container
        client_tags = Tag.SSD

    class Parameters(sdk2.Task.Parameters):

        old_atlas_to_database = sdk2.parameters.Resource('Resource with OLD atlas_to_database',
                                                         resource_type=AdfoxAtlasToDbUploader,
                                                         required=True)
        old_atlas_cache_file = sdk2.parameters.Resource('Production atlas cache file BEFORE migration',
                                                        resource_type=AdfoxAtlasCacheFile,
                                                        required=True)

        new_atlas_to_database = sdk2.parameters.Resource('Resource with NEW atlas_to_database',
                                                         resource_type=AdfoxAtlasToDbUploader,
                                                         required=True)
        new_atlas_cache_file = sdk2.parameters.Resource('Production atlas cache file AFTER migration',
                                                        resource_type=AdfoxAtlasCacheFile,
                                                        required=True)

    class Context(sdk2.Task.Context):
        has_diff = False

    @property
    def pickle_save_path(self):
        path = os.path.join(str(self.path()), 'pickle_save')
        return ensure_path(path)

    @property
    def diff_folder_path(self):
        path = os.path.join(str(self.log_path()), 'diff_folder')
        return ensure_path(path)

    def extract_atd_path(self, version):
        path = os.path.join(str(self.path()), version)
        return ensure_path(path)

    def atlas_to_database_path(self, version):
        return os.path.join(str(self.path()), version, 'atlas_to_database')

    def atlas_cache_file_path(self, version):
        if version == 'old':
            return str(sdk2.ResourceData(self.Parameters.old_atlas_cache_file).path)
        elif version == 'new':
            return str(sdk2.ResourceData(self.Parameters.new_atlas_cache_file).path)

    def atlas_to_database_tarfile_path(self, version):
        if version == 'old':
            return str(sdk2.ResourceData(self.Parameters.old_atlas_to_database).path)
        if version == 'new':
            return str(sdk2.ResourceData(self.Parameters.new_atlas_to_database).path)

    def _run_atlas_to_database(self, version):
        with tarfile.open(self.atlas_to_database_tarfile_path(version)) as tar:
            tar.extractall(path=self.extract_atd_path(version))
        with ProcessLog(self, version + 'atlas_to_database') as log:
            subprocess.check_call([self.atlas_to_database_path(version), 'to_dict',
                                   self.atlas_cache_file_path(version),
                                   version, '--save_dir', self.pickle_save_path,
                                   '--log-level', 'debug'], stdout=log.stdout, stderr=log.stderr)

    def _diff(self):
        with ProcessLog(self, 'atlas_to_database_diff') as log:
            subprocess.check_call([self.atlas_to_database_path('old'), 'diff',
                                   self.pickle_save_path, '--save_dir', self.diff_folder_path,
                                   '--log-level', 'debug'], stdout=log.stdout, stderr=log.stderr)
        return [f.split('.')[0] for f in os.listdir(self.diff_folder_path)]

    def on_execute(self):
        self._run_atlas_to_database('old')
        self._run_atlas_to_database('new')
        diff_containers = self._diff()
        if diff_containers:
            self.Context.has_diff = True
        self.set_info('has_diff: {0}, Containers with diff: {1}'.format(self.Context.has_diff, diff_containers))
