# -*- coding: utf-8 -*-

import logging
import os
import re
import shutil

from sandbox.projects.common import utils2
from sandbox import sdk2
from sandbox.projects.adfox.qa.resource_types import AdfoxQaDataFile
from sandbox.projects.adfox.resource_types import AdfoxAtlasCacheFile
from sandbox.projects.common.arcadia import sdk
from sandbox.sdk2.helpers import subprocess, ProcessLog


class AdfoxAtlasCacheGenerator(sdk2.Task):
    """
    Generates Atlas binary cache file and sqlite db with tests data. Saves them as Sandbox resources
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 12
        ram = 16 * 1024
        disk_space = 20 * 1024

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl('Arcadia Url')
        arcadia_patch = sdk2.parameters.String(
            'Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4',
            default='', multiline=True
        )

    def execute(self, logger, command, cwd='.', env=None):
        if env is None:
            env = {}
        env.update(os.environ)
        env['TZ'] = 'Europe/Moscow'
        with ProcessLog(self, logger=logger) as pl:
            retcode = subprocess.Popen(command.split(), stdout=pl.stdout, stderr=pl.stdout, cwd=cwd, env=env).wait()
            if retcode == 0:
                return
            redirect_link = utils2.resource_redirect_link(self.log_resource.id, '%s logs' % logger, pl.stdout.path.name)
            self.set_info('Task failed, see %s' % redirect_link, do_escape=False)

    def checkout_svn(self):
        arcadia_root = sdk.do_clone(self.Parameters.arcadia_url, self)
        ya_path = os.path.join(arcadia_root, 'ya')
        self.execute('checkout', ya_path + ' make --checkout adfox/amacs/tests/functional -j0 --musl', cwd=arcadia_root)
        if self.Parameters.arcadia_patch:
            sdk2.vcs.svn.Arcadia.apply_patch(arcadia_root, self.Parameters.arcadia_patch, str(self.path()))
        return os.path.join(arcadia_root, 'adfox', 'amacs', 'tests', 'functional', 'tests_amacs'), arcadia_root, ya_path

    def list_tests(self, repository_path):
        with open(os.path.join(repository_path, 'ya.make')) as f:
            test_list = f.read()
        return [x.replace(' ', '') for x in re.findall(r'RECURSE\((.*)\)', test_list, re.DOTALL)[0].split('\n') if x]

    def make_test_output_path(self, repository_path, test_name):
        test_path = os.path.join(repository_path, test_name)
        tests_list = [x[0] for x in os.walk(test_path) if
                      (os.path.split(x[0])[-1] not in ('__pycache__', 'utils')) and ('test-results' in x[1])]
        test_names = ['{}{}'.format(test_name, x.split(test_name)[-1]) for x in tests_list]
        test_results = [os.path.join(x, 'test-results') for x in tests_list]
        output = [x for dirlist in test_results for dir in os.walk(dirlist) for x in dir if 'adfox-amacs-tests' in x]
        test_output = [os.path.join(x, 'testing_out_stuff') for x in output]
        test_names = [name for name in test_names if any(name in substring for substring in test_output)]
        return zip(test_names, test_output)

    def find_atlas_segment(self, path):
        segment = filter(lambda x: x.startswith('atlas-'), os.listdir(path))
        return segment[0]

    def find_dataholder(self, path):
        data = filter(lambda x: x.startswith('data-'), os.listdir(path))
        return data[0]

    def run_tests(self, work_path, ya_path):
        cmd = '{} make -tA -r --checkout --musl -j12 --test-param make-data=1 --test-stderr -DMAKE_DATA=1'.format(
            ya_path)
        self.execute('make_data', cmd, cwd=work_path)

    def on_execute(self):
        repository_path, arcadia_root, ya_path = self.checkout_svn()
        self.run_tests(repository_path, ya_path)
        test_list = self.list_tests(repository_path)
        resource_list = []
        for test in test_list:
            path_list = self.make_test_output_path(repository_path, test)
            for name, path in path_list:
                logging.info('name:{}, path:{}'.format(name, path))
                segment_name = self.find_atlas_segment(path)
                segment_file = os.path.join(path, segment_name)
                shutil.copy(segment_file, str(self.path(segment_name)))
                resource_list.append(sdk2.ResourceData(AdfoxAtlasCacheFile(self,
                                                                           name,
                                                                           segment_name,
                                                                           target='qa.functional',
                                                                           test_name=name,
                                                                           ttl='inf')
                                                       ))

                db_name = self.find_dataholder(path)
                db_file = os.path.join(path, db_name)
                shutil.copy(db_file, str(self.path(db_name)))
                resource_list.append(sdk2.ResourceData(AdfoxQaDataFile(self,
                                                                       name,
                                                                       db_name,
                                                                       test_name=name,
                                                                       ttl='inf')
                                                       ))
        for resource in resource_list:
            resource.ready()
