# -*- coding: utf-8 -*-

import datetime as dt
import json
import os
import shutil

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.adfox.resource_types import AdfoxConfigurationRaw
from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess, ProcessLog
from sandbox.sdk2.vcs import svn

CONTAINER_WITH_MYSQL = 829148115


class AdfoxServerBuildConfiguration(sdk2.Task):
    name = "ADFOX_SERVER_BUILD_CONFIGURATION"

    class Requirements(sdk2.Task.Requirements):
        cores = 4
        disk_space = 10 * 1024
        ram = 4 * 1024
        privileged = True  # for container
        environments = (
            environments.PipEnvironment("pymysql", "0.9.3"),
        )

    class Parameters(sdk2.Task.Parameters):
        _container = sdk2.parameters.Container("LXC Container with adfox-db-5.7", default_value=CONTAINER_WITH_MYSQL,
                                               platform="linux_ubuntu_16.04_xenial", required=True)

        checkout_arcadia_from_url = sdk2.parameters.String("Get Arcadia via URL", required=False,
                                                           default_value="arcadia:/arc/trunk/arcadia")
        arcadia_patch = sdk2.parameters.String("Arcadia patch URL", required=False)

    class Context(sdk2.Task.Context):
        arc_repo_path = None
        migrator_build_task = None

    def on_prepare(self):
        # build resources
        with self.memoize_stage.build_migrator:
            path_in_arcadia = "adfox/infra/amacs_config/migrator"
            build_artifact = "config_migrator"
            self.Context.migrator_build_task = sdk2.Task["YA_MAKE_2"](
                self, owner=self.owner, description="build {0}".format(path_in_arcadia),
                checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                use_aapi_fuse=True, aapi_fallback=True,
                checkout_mode="auto", targets=path_in_arcadia,
                arts=os.path.join(path_in_arcadia, build_artifact)
            ).save().enqueue().id
        check_tasks(self, tasks=[self.Context.migrator_build_task])

        # prepare arcadia
        self.Context.arc_repo_path = svn.Arcadia.get_arcadia_src_dir(
            "{0}".format(self.Parameters.checkout_arcadia_from_url))

        # apply patch
        if self.Parameters.arcadia_patch:
            svn.Arcadia.apply_patch(
                self.Context.arc_repo_path,
                str(self.Parameters.arcadia_patch),
                str(self.path())
            )

    def on_execute(self):
        git_repo_path = os.path.join(str(self.path()), "db")
        with ProcessLog(self, "prepare_container") as log:
            subprocess.check_call(["timedatectl", "set-timezone", "Europe/Moscow"],
                                  stdout=log.stdout, stderr=log.stderr)
            subprocess.check_call(["/opt/adfox-sql/bin/virgin_mysql.sh"], stdout=log.stdout, stderr=log.stderr)
            subprocess.check_call(["git", "clone", "http://git.adfox.ru/adfox/db.git", git_repo_path],
                                  stdout=log.stdout, stderr=log.stderr)

        import pymysql
        connection = pymysql.connect(host="localhost", port=3306, user="adfox",
                                     password="", charset="cp1251")
        with connection.cursor() as cursor:
            cursor.execute("SET GLOBAL sql_mode = 'NO_ENGINE_SUBSTITUTION';")
        connection.commit()

        with ProcessLog(self, "run_migrations") as log:
            subprocess.check_call(["php", "phinx", "migrate", "-c", "migrations/sandbox_mainDB.yml"],
                                  cwd=git_repo_path, stdout=log.stdout, stderr=log.stderr)
            subprocess.check_call(["php", "phinx", "migrate", "-c", "migrations/sandbox_commonDB.yml"],
                                  cwd=git_repo_path, stdout=log.stdout, stderr=log.stderr)

        migrator_resource = sdk2.Resource.find(
            resource_type=resource_types.ARCADIA_PROJECT,
            task_id=self.Context.migrator_build_task
        ).first()
        migrator_path = os.path.join(str(sdk2.ResourceData(migrator_resource).path), "config_migrator")
        arc_repo_path = os.path.join(str(self.path()), "arc_repo")
        shutil.copytree(os.path.join(self.Context.arc_repo_path, "adfox/infra/amacs_config"), arc_repo_path)

        with ProcessLog(self, "migrator") as log:
            subprocess.check_call(
                [migrator_path, "--db-host", "localhost", "--db-user", "root",
                 "--sandbox", "migrate", "--yes"],
                cwd=os.path.join(arc_repo_path, "migrator"),
                stdout=log.stdout, stderr=log.stderr
            )

        configuration = []
        with connection.cursor(pymysql.cursors.DictCursor) as cursor:
            for ent in ["bridge", "engine", "countersd"]:
                cursor.execute("SELECT name, env, readonly, type, value "
                               "FROM adfox.config_{0} ORDER BY name DESC, env ASC".format(ent))
                configuration += cursor.fetchall()
        connection.close()

        # cast to str
        for record in configuration:
            for k, v in record.iteritems():
                if isinstance(v, (dt.datetime, dt.date, dt.time)):
                    record[k] = str(record[k])

        configuration_file = os.path.join(str(self.path()), "out.json")
        with open(configuration_file, mode="w") as out_file:
            json.dump(configuration, out_file, sort_keys=True, indent=4)

        sdk2.ResourceData(AdfoxConfigurationRaw(
            self, "json file with configuration", configuration_file
        )).ready()
