import datetime
import json
import logging
import os
import time

import requests

from sandbox import sdk2
from sandbox.projects.adfox.qa.tasks.AdfoxServerBase import AdfoxServerBase
from sandbox.projects.adfox.qa.utils.constants import YAV_ROBOT_ADFOX_SOLOMON_TOKEN, YT_PROXY
from sandbox.projects.adfox.qa.utils.secrets import get_yav_secret
from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.yql.RunYQL2 import RunYQL2
from sandbox.sandboxsdk import environments

METRICS_CHUNK = 1000
SOLOMON_URL = 'https://solomon.yandex.net/api/v2/push'


def get_yesterday():
    return (datetime.datetime.now() - datetime.timedelta(days=1)).strftime('%Y-%m-%d')


def get_ts(iso_datetime):
    return int(time.mktime(
        datetime.datetime.strptime(iso_datetime, '%Y-%m-%d %H:%M:%S').timetuple()
    ))


def normalize_label(label):
    return str(label).lower().replace(' ', '_')


# https://st.yandex-team.ru/ADFOX-17906
class AdfoxServerJstracerReports(AdfoxServerBase):
    name = 'ADFOX_SERVER_JSTRACER_REPORTS'

    class Parameters(AdfoxServerBase.Parameters):
        date_start = sdk2.parameters.String('Start date')
        date_end = sdk2.parameters.String('End date')
        output_directory = sdk2.parameters.String('Out tables directory')
        delete_existing_tables = sdk2.parameters.Bool('Delete existing tables', default=True)

    class Requirements(AdfoxServerBase.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('requests'),
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('psycopg2-binary', '2.7.7'),
        )

    class Context(sdk2.Task.Context):
        tasks = []

    def on_save(self):
        if not self.Parameters.date_start:
            self.Parameters.date_start = get_yesterday()
        if not self.Parameters.date_end:
            self.Parameters.date_end = get_yesterday()
        if not self.Parameters.output_directory:
            self.Parameters.output_directory = '//home/adfox/jstracer'

    def start_request(self, tag, date, **query_args):

        output_table = '{0}/{1}/{2}'.format(self.Parameters.output_directory, tag, date)
        self.Context.tasks.append({
            'output_table': output_table,
            'tag': tag
        })

        if self.yt.exists(output_table):
            if self.Parameters.delete_existing_tables:
                self.yt.remove(output_table)
            else:
                return

        query_args.update({
            "#JS_TRACER_TABLE#": "{0}.`//logs/jstracer-log/1d/{1}`".format(YT_PROXY, date),
            "#OUTPUT_TABLE#": "{0}.`{1}`".format(YT_PROXY, output_table)
        })

        dir_path = os.path.dirname(os.path.realpath(__file__))
        with open(os.path.join(dir_path, 'functions.yql'), 'r') as functions_file:
            functions = functions_file.read()
        with open(os.path.join(dir_path, '{0}.yql'.format(tag)), 'r') as query_file:
            query = functions + query_file.read()
        for k, v in query_args.items():
            query = query.replace(k, v)

        task = RunYQL2(self, description='Calculating {0}'.format(tag), owner=self.owner)

        task.fail_on_any_error = True
        task.Parameters.query = query
        task.Parameters.publish_query = True
        task.Parameters.use_v1_syntax = True
        task.Parameters.trace_query = True
        task.Parameters.retry_period = 300

        task.save().enqueue()

    def process_response(self, task_info):
        solomon_token = get_yav_secret(YAV_ROBOT_ADFOX_SOLOMON_TOKEN)['SOLOMON_TOKEN']

        stats = list(self.yt.read_table(task_info["output_table"]))
        if not stats:
            logging.error('Zero rows in table: {0}'.format(task_info["output_table"]))
            return

        fields = stats[0].keys()
        percentiles = [f for f in fields if f.startswith('P') and f[1:].isdigit()]
        has_hits = 'Hits' in fields
        labels = [f for f in fields if f not in percentiles + ["Hits", "HourTime"]]

        chunks = [stats[i:i + METRICS_CHUNK] for i in range(0, len(stats), METRICS_CHUNK)]
        for chunk in chunks:
            request_body = {
                'commonLabels': {
                    'host': 'sandbox'
                },
                'sensors': []
            }
            for stat in chunk:
                # percentiles
                for percentile in percentiles:
                    labels_values = {k: stat[k] for k in labels}
                    labels_values.update({
                        "percentile": percentile,
                        "tag": task_info["tag"]
                    })
                    request_body['sensors'].append({
                        'labels': {normalize_label(k): normalize_label(v) for k, v in labels_values.items()},
                        'ts': get_ts(stat['HourTime']),
                        'value': stat[percentile]
                    })
                # hits
                if has_hits:
                    labels_values = {k: stat[k] for k in labels}
                    labels_values.update({
                        "tag": task_info["tag"]
                    })
                    request_body['sensors'].append({
                        'labels': {normalize_label(k): normalize_label(v) for k, v in labels_values.items()},
                        'ts': get_ts(stat['HourTime']),
                        'value': stat['Hits']
                    })
            response = requests.post(
                SOLOMON_URL,
                params={
                    'project': 'adfox',
                    'cluster': 'production',
                    'service': 'jstracer'
                },
                headers={
                    'Content-Type': 'application/json',
                    'Authorization': 'OAuth {0}'.format(solomon_token)
                },
                data=json.dumps(request_body))
            logging.info(response.content)

    def on_execute(self):

        with self.memoize_stage.query(commit_on_entrance=False):
            date_start = datetime.datetime.strptime(self.Parameters.date_start, '%Y-%m-%d')
            date_end = datetime.datetime.strptime(self.Parameters.date_end, '%Y-%m-%d')

            while date_start.date() <= date_end.date():
                date_start_str = date_start.strftime('%Y-%m-%d')
                self.start_request('performance_adfox', date_start_str)
                self.start_request('block_render_adfox', date_start_str)
                self.start_request('performance_pcode_context', date_start_str)
                self.start_request('render_marks_pcode_errors', date_start_str)
                date_start += datetime.timedelta(days=1)

        tasks = [t for t in self.find(RunYQL2)]
        check_tasks(self, tasks)

        for task_info in self.Context.tasks:
            self.process_response(task_info)
