# -*- coding: utf-8 -*-

import json
import logging
import sys
from os.path import join as join_path, realpath, dirname

from sandbox import sdk2
from sandbox.projects.common.juggler import jclient
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.process import run_process

DEFAULT_JUGGLER_HOST = 'adfox.sandbox.yandex-team.ru'
DEFAULT_JUGGLER_SERVICE = 'adfox-reports-backup-monitoring'

DEFAULT_MYSQL_HOST = 'bkp1.sas.adfox.ru'
DEFAULT_MYSQL_USER = 'sandbox-check'
DEFAULT_MYSQL_PASS_VAULT = 'ADFOX_REPORT_BACKUP_MYSQL_PASS'
DEFAULT_MYSQL_DB = 'reports_adfox'


class CheckFailed(Exception):
    pass


class AdfoxReportMonitoring(sdk2.Task):
    """ Проверяем бэкап базы отчетов """

    class Parameters(sdk2.Task.Parameters):
        juggler_host = sdk2.parameters.String('Juggler host', default=DEFAULT_JUGGLER_HOST, required=True)
        juggler_service = sdk2.parameters.String(
            'Juggler service',
            default=DEFAULT_JUGGLER_SERVICE,
            required=True
        )

        with sdk2.parameters.Group("mysql") as db_params:
            mysql_host = sdk2.parameters.String('Database host', default=DEFAULT_MYSQL_HOST, required=True)
            mysql_user = sdk2.parameters.String('Database user', default=DEFAULT_MYSQL_USER, required=True)
            mysql_password_vault = sdk2.parameters.String(
                'Database password vault name',
                default=DEFAULT_MYSQL_PASS_VAULT,
                required=True
            )
            mysql_reports_db = sdk2.parameters.String('Reports db', default=DEFAULT_MYSQL_DB, required=True)

    def on_execute(self):
        logging.getLogger().setLevel(logging.DEBUG)
        handler = logging.StreamHandler(sys.stdout)
        handler.setLevel(logging.DEBUG)
        formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
        handler.setFormatter(formatter)
        logging.getLogger().addHandler(handler)
        logging.info('Starting script reports_monitoring...')

        script_path = join_path(dirname(realpath(__file__)), 'reports_monitoring.py')

        run_success = None

        with environments.VirtualEnvironment() as venv:
            environments.PipEnvironment('pymysql', venv=venv, use_wheel=True, version='0.9.3').prepare()
            mysql = {
                'host': self.Parameters.mysql_host,
                'user': self.Parameters.mysql_user,
                'password': sdk2.Vault.data('ADFOX', self.Parameters.mysql_password_vault),
                'db': self.Parameters.mysql_reports_db,
            }
            try:
                run_process([
                    venv.executable,
                    script_path,
                    "--mysql='{}'".format(json.dumps(mysql)),
                ], log_prefix='reports_monitoring', shell=True)
                run_success = True
            except SandboxSubprocessError:
                run_success = False

        if run_success is True:
            logging.info('Backup check finished successful!')
            self.notify('OK', 'Backup is OK!')
        else:
            logging.error('Backup check failed!')
            self.notify(
                'CRIT',
                'Backup check FAILED! https://wiki.yandex-team.ru/users/forx/Bjekap-otchjotov-do-2016-goda/'
            )
            raise CheckFailed

    def notify(self, status, message):
        jclient.send_events_to_juggler(
            self.Parameters.juggler_host,
            self.Parameters.juggler_service,
            status,
            message
        )
