# -*- coding: utf-8 -*-
import os
import tarfile
import json
import logging
from datetime import datetime, timedelta
from dateutil import parser

import sandbox.common.types.task as ctt
from sandbox.projects.common import utils2
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.common.errors import TaskFailure
from sandbox.projects.adfox.resource_types import AdfoxUnloadClientLogsVh
from sandbox.sdk2.helpers import ProcessLog, ProcessRegistry, subprocess
from sandbox.projects.common.juggler import jclient

SANDBOX_VAULT_OWNER = 'ADFOX'
SB_ROBOT_ADFOX_NIRVANA_TOKEN = 'ROBOT_ADFOX_NIRVANA_TOKEN'

DEFAULT_YT_TOKEN = 'ADFOX_ROBOT_YT_TOKEN'
DEFAULT_YT_OWNER = 'ADFOX'

DEFAULT_CH_HOST = 'ch.mdb.adfox.net'
DEFAULT_CH_PORT = '9440'
DEFAULT_CH_USER = 'dump'
DEFAULT_CH_PASSWORD_VAULT_NAME = 'adfox_mdb_ch_dump_password'

REPEAT_SLEEP = 1  # seconds
MAX_WAIT_RESULT = 30  # seconds

LOGGER = 'vh_graph_log'
HORIZON_WAIT_TIME = 720  # minutes minimum diff between horizon and end of hour/day to start unload
WAIT_TIME = 720  # minutes wait after hour/day finished before beginning unload

JUGGLER_HOST = 'adfox.sandbox.yandex-team.ru'

JUGGLER_DELAY_HOURS = 5
JUGGLER_DELAY_DAYS = 1


class AdfoxUnloadClientLogs(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        environments = (
            environments.PipEnvironment('python-jsonrpc'),
            environments.PipEnvironment('yandex-yt', '0.8.17-0'),
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10800  # 3h

        with sdk2.parameters.Group('Report generate settings') as report_settings:
            unload_vh_executable = sdk2.parameters.LastReleasedResource(
                'AdfoxUnloadClientLogsVh executable that creates & run valhalla graph',
                required=True, resource_type=AdfoxUnloadClientLogsVh)

        with sdk2.parameters.Group("YT Parameters") as yt_block:
            yt_token_vault_name = sdk2.parameters.String('Vault name to extract YT token', default=DEFAULT_YT_TOKEN,
                                                         required=True)
            yt_token_vault_owner = sdk2.parameters.String('Vault owner to extract YT token', default=DEFAULT_YT_OWNER,
                                                          required=True)
            yt_path = sdk2.parameters.String(
                'locke path to clients settings',
                default="//home/adfox/unload_client_logs",
                required=True,
            )

        with sdk2.parameters.Group("ClickHouse Parameters") as ch_block:
            ch_host_name = sdk2.parameters.String('host name', default=DEFAULT_CH_HOST, required=True)
            ch_host_port = sdk2.parameters.String('host port', default=DEFAULT_CH_PORT, required=True)
            ch_user_name = sdk2.parameters.String('user name', default=DEFAULT_CH_USER, required=True)
            ch_user_password_path = sdk2.parameters.String('user pass path in nirvana vault',
                                                           default=DEFAULT_CH_PASSWORD_VAULT_NAME, required=True)
        with sdk2.parameters.Group("rtd") as rtd_params:
            rtd_host = sdk2.parameters.String('RTD host', default='http://rtd.adfox.ru/rpc', required=True)

        with sdk2.parameters.Group("Report Parameters") as report_block:
            report_name = sdk2.parameters.String('report name', default='', required=True)
            take_date_from_locke = sdk2.parameters.Bool("Take date from locke data for this type of report",
                                                        default=True, required=True)
            with take_date_from_locke.value[False]:
                report_date = sdk2.parameters.String('Date to build report', required=True)

    def on_enqueue(self):
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='Adfox_Unload_Client_Logs_{}'.format(self.Parameters.report_name),
                                       weight=1, capacity=1)
            ],
        )

    def on_execute(self):
        from yt.wrapper import YtClient
        yt_token = sdk2.Vault.data(self.Parameters.yt_token_vault_owner, self.Parameters.yt_token_vault_name)
        yt_client = YtClient(proxy='locke', token=yt_token)

        current_path = str(self.path())
        unload_vh_tar_path = str(sdk2.ResourceData(self.Parameters.unload_vh_executable).path)
        with tarfile.open(unload_vh_tar_path) as tar:
            tar.extractall(path=current_path)
        unload_vh_path = os.path.join(current_path, 'unload_client_logs')
        env = dict(os.environ,
                   NIRVANA_TOKEN=sdk2.Vault.data(SANDBOX_VAULT_OWNER, SB_ROBOT_ADFOX_NIRVANA_TOKEN),
                   YT_TOKEN=yt_token)

        report_to_unload_data = json.loads(
            yt_client.get('{}/{}'.format(self.Parameters.yt_path, self.Parameters.report_name)))

        if report_to_unload_data['type'] == 'hourly':
            report_step = timedelta(hours=1)
        elif report_to_unload_data['type'] == 'daily':
            report_step = timedelta(days=1)
        else:
            raise Exception('unknown report type in locke record for {} report'.format(self.Parameters.report_name))

        # report_date - timestamp of report start, so report_date = "2020-01-01 15:00:00" means
        # that it contains records from 15:00 to 16:00 (in case of hourly report)
        # in locke we store a timestamp of a last successfully calculated report,
        # so we need to add a report_step to get a timestamp for a new report (that will be calculated now)
        # if we want to start report calculation,
        # we need to ensure that datetime.now() > report_date + report_step + WAIT_TIME
        if self.Parameters.take_date_from_locke:
            report_date = datetime.strptime(report_to_unload_data['date'], "%Y-%m-%d %H:%M:%S") + report_step
        else:
            report_date = parser.parse(self.Parameters.report_date)
        logging.info('Running clients logs unloader {report} reportd for {date} date'.format(
            report=self.Parameters.report_name, date=report_date))

        if datetime.now() < report_date + report_step + timedelta(minutes=WAIT_TIME):
            logging.info('cant unload this day/hour, should wait {} minutes after it ends before unload'.format(WAIT_TIME))
        elif self.__check_event_horizon(report_date=report_date, report_step=report_step):
            with ProcessRegistry, ProcessLog(self, logger=LOGGER) as pl:
                options = [
                    '--ch-host', self.Parameters.ch_host_name,
                    '--ch-port', self.Parameters.ch_host_port,
                    '--ch-user', self.Parameters.ch_user_name,
                    '--ch-pass', self.Parameters.ch_user_password_path,
                    '--access-key-id', report_to_unload_data['aws_id'],
                    '--secret-access-key', report_to_unload_data['aws_key'],
                    '--bucket', report_to_unload_data['mds_bucket'],
                    '--folder', report_to_unload_data['mds_folder'],
                    '--report-date', report_date.strftime("%Y-%m-%d %H:%M:%S"),
                    '--report-name', self.Parameters.report_name,
                ]
                logging.info('Running graph')
                return_code = subprocess.Popen([unload_vh_path] + options,
                                               stdout=pl.stdout, stderr=pl.stderr, env=env).wait()
                if return_code != 0:
                    redirect_link = utils2.resource_redirect_link(
                        self.log_resource.id, '%s logs' % LOGGER, pl.stderr.path.name
                    )
                    self.set_info('Task failed, see %s' % redirect_link, do_escape=False)
                    raise TaskFailure('%s failed' % LOGGER)
            if self.Parameters.take_date_from_locke:
                report_to_unload_data = json.loads(yt_client.get('{}/{}'.format(self.Parameters.yt_path, self.Parameters.report_name)))
                report_to_unload_data['date'] = report_date.strftime("%Y-%m-%d %H:%M:%S")
                report_to_unload_data['last_update'] = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
                yt_client.set('{}/{}'.format(self.Parameters.yt_path, self.Parameters.report_name),
                              json.dumps(report_to_unload_data))

                # write to juggler only if unload date/hour is more then current time minus maximum delay
                # so if we unload too old report - no notification to juggler will be sent
                if (report_to_unload_data['type'] == 'hourly'
                    and datetime.now() < report_date + timedelta(hours=JUGGLER_DELAY_HOURS) + report_step) \
                        or (report_to_unload_data['type'] == 'daily'
                            and datetime.now() < report_date + timedelta(days=JUGGLER_DELAY_DAYS) + report_step):
                    jclient.send_events_to_juggler(
                        JUGGLER_HOST,
                        'ADFOX_UNLOAD_CLIENT_LOGS_{}'.format(self.Parameters.report_name),
                        'OK',
                        'Script has been successfully completed.'
                    )
        else:
            logging.info('event horizon is not reached')
        logging.info('Exiting task')

    def __check_event_horizon(self, report_date, report_step):
        from get_event_horizon import GetEventHorizon
        params = {
            'host': self.Parameters.rtd_host,
            'max_wait_result': MAX_WAIT_RESULT,
            'repeat_check': REPEAT_SLEEP,
        }
        horizon = GetEventHorizon(params).get_event_horizon()
        logging.info('event horizon is {}'.format(horizon.strftime("%Y-%m-%d %H:%M:%S")))
        if report_date + report_step > horizon - timedelta(minutes=HORIZON_WAIT_TIME):
            logging.info('event horizon is not reached')
            return False
        return True
