# -*- coding: utf-8 -*-
import json
import logging
import random
import time
import requests
from datetime import datetime

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.adfox.resource_types import AdfoxUnloadClientLogsVh
from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.adfox.unload_client_logs import AdfoxUnloadClientLogs

SANDBOX_VAULT_OWNER = 'ADFOX'
SB_ROBOT_ADFOX_NIRVANA_TOKEN = 'ROBOT_ADFOX_NIRVANA_TOKEN'

DEFAULT_YT_TOKEN = 'ADFOX_ROBOT_YT_TOKEN'
DEFAULT_YT_OWNER = 'ADFOX'

DEFAULT_CH_HOST = 'ch.mdb.adfox.net'
DEFAULT_CH_PORT = '9440'
DEFAULT_CH_USER = 'dump'
DEFAULT_CH_PASSWORD_VAULT_NAME = 'adfox_mdb_ch_dump_password'

LOCKE_MDB_HOSTS_PATH = '//home/adfox/mdb_clickhouse/hosts'


class AdfoxUnloadClientLogsScheduler(sdk2.Task):

    class Context(sdk2.Task.Context):
        unload_task_id = None       # Идентификатор подзадачи

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        environments = (
            environments.PipEnvironment('yandex-yt', '0.8.17-0'),
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10800  # 3h

        with sdk2.parameters.Group('Report generate settings') as report_settings:
            unload_vh_executable = sdk2.parameters.LastReleasedResource(
                'AdfoxUnloadClientLogsVh executable that creates & run valhalla graph',
                required=True, resource_type=AdfoxUnloadClientLogsVh)

        with sdk2.parameters.Group("YT Parameters") as yt_block:
            yt_token_vault_name = sdk2.parameters.String('Vault name to extract YT token', default=DEFAULT_YT_TOKEN,
                                                         required=True)
            yt_token_vault_owner = sdk2.parameters.String('Vault owner to extract YT token', default=DEFAULT_YT_OWNER,
                                                          required=True)
            yt_path = sdk2.parameters.String('locke path to clients settings', default="//home/adfox/unload_client_logs",
                                             required=True)

        with sdk2.parameters.Group("ClickHouse Parameters") as ch_block:
            take_random_mdb_host = sdk2.parameters.Bool("Take random mdb host from locke list", default=True,
                                                        required=True)
            with take_random_mdb_host.value[False]:
                ch_host_name = sdk2.parameters.String('host name', default=DEFAULT_CH_HOST, required=True)
                ch_host_port = sdk2.parameters.String('host port', default=DEFAULT_CH_PORT, required=True)
            ch_user_name = sdk2.parameters.String('user name', default=DEFAULT_CH_USER, required=True)
            ch_user_password_path = sdk2.parameters.String('user pass path in nirvana vault',
                                                           default=DEFAULT_CH_PASSWORD_VAULT_NAME, required=True)
        with sdk2.parameters.Group("rtd") as rtd_params:
            rtd_host = sdk2.parameters.String('RTD host', default='http://rtd.adfox.ru/rpc', required=True)

    def on_execute(self):
        logging.info('Running clients logs unloader')
        with self.memoize_stage["run_unload"]:
            self.proceed()
        check_tasks(self, tasks=[self.Context.unload_task_id])
        logging.info('Unload finished successfully')
        logging.info('Exiting task')

    def proceed(self):
        from yt.wrapper import YtClient, ypath_join
        yt_client = YtClient(proxy='locke', token=sdk2.Vault.data(
            self.Parameters.yt_token_vault_owner, self.Parameters.yt_token_vault_name))

        # read reports list
        reports_list = yt_client.list(self.Parameters.yt_path)
        reports_data = {}
        for k in reports_list:
            reports_data[k] = json.loads(yt_client.get(ypath_join(self.Parameters.yt_path, k)))

        # select report - scheduler finds oldest report in locke and tries to unload it
        for k, v in sorted(reports_data.items(), key=lambda item: item[1]['last_update'], reverse=False):
            report_name = k
            break
        report_to_unload_data = reports_data[report_name]

        # we should update last_update for this report to indicate that we (at least) tried to launch unload for it
        report_to_unload_data['last_update'] = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        yt_client.set('{}/{}'.format(self.Parameters.yt_path, report_name), json.dumps(report_to_unload_data))

        if self.Parameters.take_random_mdb_host:
            ch_host = self.choose_mdb_host(yt_client)
            ch_port = DEFAULT_CH_PORT
        else:
            ch_host = self.Parameters.ch_host_name
            ch_port = self.Parameters.ch_host_port
        logging.info('using ch host {}:{}'.format(ch_host, ch_port))

        logging.info('Launching unload subtask...')
        self.Context.unload_task_id = self.__launch_unloader(
            report_name=report_name, ch_host=ch_host, ch_port=ch_port).id
        logging.info("Waiting for subtask {}".format(self.Context.unload_task_id))

    def choose_mdb_host(self, yt_client):
        hosts = json.loads(yt_client.get(LOCKE_MDB_HOSTS_PATH))
        hosts = [h for h in hosts if not h.startswith("man-")]
        random.seed(time.time())
        random.shuffle(hosts)
        for host in hosts[0:10]:
            r = requests.get('https://{}:8443'.format(host), verify='/etc/ssl/certs/ca-certificates.crt')
            if r.content.strip() == "Ok.":
                return host

        raise Exception('cant find any alive ch host in first 10 random hosts from locke record')

    def __launch_unloader(self, report_name, ch_host, ch_port, report_date=None):
        logging.info("Starting unload client logs subtask {} report"
                     .format(report_name))

        logging.info("Creating subtask")
        subtask = AdfoxUnloadClientLogs(
            self,
            description="unload {} report".format(report_name),
            owner=self.owner,
        )

        subtask.fail_on_any_error = True

        subtask.Parameters.yt_token_vault_name = self.Parameters.yt_token_vault_name
        subtask.Parameters.yt_token_vault_owner = self.Parameters.yt_token_vault_owner
        subtask.Parameters.yt_path = self.Parameters.yt_path

        subtask.Parameters.ch_host_name = ch_host
        subtask.Parameters.ch_host_port = ch_port
        subtask.Parameters.ch_user_name = self.Parameters.ch_user_name
        subtask.Parameters.ch_user_password_path = self.Parameters.ch_user_password_path

        subtask.Parameters.unload_vh_executable = self.Parameters.unload_vh_executable
        subtask.Parameters.report_name = report_name
        if report_date:
            subtask.Parameters.take_date_from_locke = False
            subtask.Parameters.report_date = report_date
        else:
            subtask.Parameters.take_date_from_locke = True

        subtask.Parameters.rtd_host = self.Parameters.rtd_host

        subtask.save()
        subtask.enqueue()
        return subtask
