# -*- coding: utf-8 -*-

import logging

from sandbox.sandboxsdk import environments
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.common.utils import chain

from sandbox import sdk2


class AdfoxYtLogChecker(sdk2.Task):
    class Context(sdk2.Task.Context):
        yql_check_task_id = None       # Идентификатор подзадачи

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("YT Parameters") as yt_block:
            yt_token_vault_name = sdk2.parameters.String('Vault name to extract YT token', default='ADFOX_ROBOT_YT_TOKEN', required=True)
            yt_token_vault_owner = sdk2.parameters.String('Vault owner to extract YT token', default='ADFOX', required=True)
            yt_database = sdk2.parameters.String('YT database name', default="HAHN", required=True)
            yt_logs_path = sdk2.parameters.String('Path to hourly logs', default='//home/logfeller/logs/adfox-event-log/1h', required=True)

        with sdk2.parameters.Group("YT Parameters") as check_block:
            normal_lag_minutes = sdk2.parameters.Integer('Normal lag for hourly logs (minutes)', default=4*60, required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt', '0.8.17-0'),
        )

    def __get_yt_token(self):
        return sdk2.Vault.data(self.Parameters.yt_token_vault_owner, self.Parameters.yt_token_vault_name)

    def __get_yt_client(self):
        import yt.wrapper as yt

        return yt.YtClient(proxy=self.Parameters.yt_database, token=self.__get_yt_token())

    def __get_available_log_table(self):
        attr_row_count = 'row_count'

        cli = self.__get_yt_client()
        lst_tables = cli.list(self.Parameters.yt_logs_path, attributes=[attr_row_count])
        lst_tables = filter(
            lambda item: item.attributes[attr_row_count] > 0,
            lst_tables
        )
        if not len(lst_tables):
            raise TaskFailure('There is no available log table - no tables at all, or they are empty')

        lst_tables.sort(reverse=True)
        return str(lst_tables[0])

    def __check_logs_lag(self, latest_table_name):
        from datetime import datetime

        logs_date_time = datetime.strptime(latest_table_name, '%Y-%m-%dT%H:%M:%S')
        dt_now = datetime.now()
        diff = (dt_now - logs_date_time).total_seconds() // 60
        logging.info('Latest table is [{}]. Lag is {} minutes'.format(latest_table_name, diff))

        if diff > self.Parameters.normal_lag_minutes:
            raise TaskFailure('Hourly logs lag is too long - {}'.format(diff))

    def __launch_yql_query(self, latest_table_name):
        from projects.yql.RunYQL2 import RunYQL2

        str_test_query = """
            USE {cluster};

            SELECT *
            FROM `//home/logfeller/logs/adfox-event-log/1h/{table_name}`
            LIMIT 1;
        """

        subtask = RunYQL2(
            self,
            description="Run test YQL query over hourly logs table (subtask for {})".format(self.id),
            owner=self.owner,  # required fields cannot be empty
        )

        subtask.fail_on_any_error = True
        subtask.Parameters.query = str_test_query.format(
            table_name=latest_table_name,
            cluster=self.Parameters.yt_database,
        )

        subtask.Parameters.trace_query = True
        subtask.Parameters.use_v1_syntax = True

        logging.info("Launching query subtask")
        subtask.save()
        subtask.enqueue()

        return subtask

    def on_execute(self):
        logging.info('Sarted check hourly logs task')

        with self.memoize_stage["run_checks"]:
            logging.info('Asking for latest hourly log table')
            latest_log_table = self.__get_available_log_table()

            logging.info('Checking for hourly logs lag...')
            self.__check_logs_lag(latest_log_table)

            logging.info('Launching test query...')
            self.Context.yql_check_task_id = self.__launch_yql_query(latest_log_table).id
            logging.info("Waiting for subtasks [{}]".format(self.Context.yql_check_task_id))
            raise sdk2.WaitTask(
                [self.Context.yql_check_task_id],
                set(chain(Status.Group.FINISH, Status.Group.BREAK)),
                wait_all=True
            )

        with self.memoize_stage["finalize_checks"]:
            logging.info('There wes launched subtask - {}. Checking it'.format(str(self.Context.yql_check_task_id)))
            if not self.find(id=self.Context.yql_check_task_id, status=Status.SUCCESS).count:
                raise TaskFailure('YQL test run has been failed')
            logging.info('All subtasks finished successfully')

        logging.info('Exiting task')
