# -*- coding: utf8

import enum

from sandbox import sdk2

VALID_OWNER = "ADS-ML-STORAGE"
RELEASERS = ["robot-ml-logos", "robot-logos", "ohmikey", "dim-gonch", "ilariia", "maksimkulis", "ernurator"]
BS_RELEASERS = ["gmikee"]


class EMlStorageClientMode(enum.Enum):
    PROD = "production"  # использует релизнутую версию бинаря для работы со стораджем
    DEV = "development"  # использует транковую версию бинаря для работы со стораджем
    TEST = "testing"     # работает полностью в локальном режиме


class EMlStorageEnv(enum.Enum):
    PROD = "production"        # Среда для продашкшена. Модели будут интерпретироваться транспортом
    DEV = "development"        # Среда для тестовых запусков. Модели не будут интерпретироваться транспортом


class EMlStorageUploadStatus(enum.Enum):
    READY = "ready"            # Модель успешно загружена со всеми компоненами
    NOT_READY = "not_ready"    # Модель не загружена в сторадж до конца (например, загрузился только один кусок дампа)
    BROKEN = "broken"          # Модель сломана при загрузке
    DELETING = "deleting"      # Модель в процессе удаления
    NO_DUMPS = "no_dumps"      # Модель удалена, остался только дамп меты
    DELETED = "deleted"        # Модель удалена


class EMlStorageTransportStatus(enum.Enum):
    DRAFT = "draft"            # Свежая модель, которую еще не обработал транспорт
    NEW = "new"                # Модель, требующая валидации
    VALID = "valid"            # Модель, успешно прошедшая валидацию
    INVALID = "invalid"        # Модель, не прошедшая валидацию
    REJECTED = "rejected"      # Модель, у которой валидация запускаться не будет


class MlStorageHead(sdk2.Resource):
    """Ml Storage head resource"""
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    ######################
    # Продуктовые атрибуты
    ######################

    # Тип модели в сторадже, например, "vw_apc_model"
    key = sdk2.parameters.String("Key", required=True)
    # Версия модели. Ожидается дата последнего лога используемого при обучении в ISO. Например, "2021-01-31T14:00:00"
    version = sdk2.parameters.String("Version", required=True)

    ###########################
    # Инфраструктурные атрибуты
    ###########################

    # Провязка с дампами, например {'meta': 123, 'page': 345, 'banner': 678}
    dump_resource_ids = sdk2.parameters.String("Resource ids of dumps", default_value="{}")
    # Позволяет отличить тестовые запуски от продовых. См. EMlStorageEnv
    environment_type = sdk2.parameters.String(
        "Environment",
        default_value=EMlStorageEnv.DEV.value,
        required=True
    )
    # Позволяет отличать статусы загрузки модели в сторадж. См. EMlStorageUploadStatus
    upload_status = sdk2.parameters.String(
        "Upload status",
        default_value=EMlStorageUploadStatus.NOT_READY.value,
        required=True
    )
    # Позволяет найти id загрузки при перезаливке
    upload_resource_id = sdk2.parameters.Integer("Upload resource_id", default_value=0)
    # Позволяет отличать статусы обработки транспортом. См. EMlStorageTransportStatus
    transport_status = sdk2.parameters.String(
        "Transport status",
        default_value=EMlStorageTransportStatus.DRAFT.value,
        required=True
    )
    # Позволяет управлять статусом транспорта
    # Пока не используется
    force_transport_status = sdk2.parameters.String("Force transport status", default_value=None, required=True)
    # Позволяет отличать продовые модели от непродовых. Выставляется вначале у дампов, потом у head
    # Пока не используется
    is_prod = sdk2.parameters.Bool("Is model prod", default_value=False, required=True)
    # Позволяет управлять статусом прода
    # Пока не используется
    force_prod = sdk2.parameters.Bool("Force model prod", default_value=False, required=True)


class MlStorageDump(sdk2.Resource):
    """Ml Storage dump resource"""
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    ######################
    # Продуктовые атрибуты
    ######################

    # Тип модели в сторадже, например, "vw_apc_model"
    key = sdk2.parameters.String("Key", required=True)
    # Версия модели. Ожидается дата последнего лога используемого при обучении в ISO. Например, "2021-01-31T14:00:00"
    version = sdk2.parameters.String("Version", required=True)
    # Айдишник кусочка модели в реестре, используется в для заведения экспериментов в проде
    registry_id = sdk2.parameters.Integer("Model dump ID in registry")

    ###########################
    # Инфраструктурные атрибуты
    ###########################

    # Ключ, по которому head провязан с дампом. Например, "page". См. MlStorageHead.dump_resource_ids
    dump_key = sdk2.parameters.String("Dump identifier", required=True)
    # Позволяет найти head этой модели
    head_resource_id = sdk2.parameters.Integer("Identifier of head resource", required=True)
    # Позволяет отличить тестовые запуски от продовых. См. EMlStorageEnv
    environment_type = sdk2.parameters.String(
        "Environment",
        default_value=EMlStorageEnv.DEV.value,
        required=True
    )
    # Позволяет отличать продовые модели он непродовых при запросах на дампы. Берется последний VALID
    # Будет заменен на is_prod или binary pull
    transport_status = sdk2.parameters.String(
        "Transport status",
        default_value=EMlStorageTransportStatus.DRAFT.value,
        required=True
    )
    # Позволяет отличать продовые модели от непродовых при запросах на дампы
    # Пока не используется
    # Будет заменен на binary pull
    is_prod = sdk2.parameters.Bool("Is model prod", default_value=False, required=True)


# Используется для валидации того, что не переписываются сервисные аттрибуты при добавлении кастомных атрибутов
MlStorageServiceAttrs = {key for key in list(MlStorageHead.__dict__) + list(MlStorageDump.__dict__) if not key.startswith("_")}


# ML Storage Config Resources

class MlStorageValidConfig(sdk2.Resource):
    """Ml Storage resource with json of all models"""
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    releasable = True
    releasers = RELEASERS
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStorageDiffReport(sdk2.Resource):
    """Ml Storage Diff Report"""
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    releasable = True
    releasers = RELEASERS + BS_RELEASERS
    report_type = sdk2.parameters.String("Report type", default_value="common", required=True)


# ML Storage Linear Model Configs

class MlStorageLmConfig(sdk2.Resource):
    """Ml Storage LinearModelConfig"""
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    releasable = True
    releasers = RELEASERS + BS_RELEASERS
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStorageLmScheme(sdk2.Resource):
    """Ml Storage LinearModelScheme"""
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    releasable = True
    releasers = RELEASERS + BS_RELEASERS
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


# ML Storage Binary Resources

class MlStorageCliBinary(sdk2.Resource):
    """ Executable Client Binary for ML Storage """
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS

    arcadia_target = "ads/emily/storage/models/client/bin/ml-storage-cli"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStoragePreprocessorMlEngineBinary(sdk2.Resource):
    """ Executable Preprocessor ML Engine Binary for ML Storage """
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS

    arcadia_target = "ads/emily/storage/libs/nirvana/blocks/preprocessor/bin/ml-storage-preprocessor"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStorageIndexerBinary(sdk2.Resource):
    """ Executable Indexer Binary for ML Storage """
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS

    arcadia_target = "ads/emily/storage/transport/indexer/bin/indexer"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStorageReactorBinary(sdk2.Resource):
    """ Executable Reactor Binary for ML Storage """
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS

    arcadia_target = "ads/emily/storage/transport/indexer/reactor/bin/reactor"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStorageModelProcessorBinary(sdk2.Resource):
    """ Executable Model Processor Binary for ML Storage """
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS

    arcadia_target = "ads/emily/storage/libs/nirvana/blocks/processor/bin/processor"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStorageIndexerPreprodBinary(sdk2.Resource):
    """ Executable Indexer Preprod Binary for ML Storage """
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS

    arcadia_target = "ads/emily/storage/transport/indexer/preprod/bin/indexer_preprod"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStorageYtLoaderBinary(sdk2.Resource):
    """ Executable YT Lodader Binary for ML Storage """
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releaseable = True
    releasers = RELEASERS + BS_RELEASERS

    arcadia_target = "ads/emily/storage/transport/yt_loader/bin/yt_loader"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStorageBsConfigGenerator(sdk2.Resource):
    """Ml Storage Bs config Generator"""
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS

    arcadia_target = "ads/emily/storage/models/config/generators/bs_prod/bin/generate"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStorageTsarsConfigExporter(sdk2.Resource):
    """Ml Storage Tsars Config Exporter"""
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS
    arcadia_target = "ads/emily/storage/models/config/generators/config_exporters/bin/exporter"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStorageMonitorBinary(sdk2.Resource):
    """ Executable Monitor Binary for ML Storage """
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS

    arcadia_target = "ads/emily/storage/libs/monitor/bin/emily-monitor"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStorageSoloBinary(sdk2.Resource):
    """ Executable Solo Binary for ML Storage """
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS

    arcadia_target = "ads/emily/storage/libs/monitor/alerts/solo/creator/solo"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStorageCleanerBinary(sdk2.Resource):
    """ Executable Clean Binary for ML Storage """
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS

    arcadia_target = "ads/emily/storage/scripts/clean/clean"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


# ML Storage Validators

class MlStorageDummyValidatorBinary(sdk2.Resource):
    """ Executable Dummy Validator for ML Storage """
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS

    arcadia_target = "ads/emily/storage/transport/indexer/validators/dummy/validator"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStorageCatboostValidatorBinary(sdk2.Resource):
    """ Executable Catboost Validator for ML Storage """
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS

    arcadia_target = "ads/emily/storage/transport/indexer/validators/catboost_validator/validator"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlStoragePytorchValidatorBinary(sdk2.Resource):
    """ Executable Pytorch Validator for ML Storage """
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS + ["alxmopo3ov", "robdrynkin", "vzaborovskiy", "jruziev"]

    arcadia_target = "ads/emily/storage/transport/indexer/validators/pytorch/bin/validator"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


# ML Logos Viewer Resources

class MlLogosViewerData(sdk2.Resource):
    """ Ml Logos Viewer Data """
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)
    dump_id = sdk2.parameters.String("Dump identifier", default=None)
    any_arch = True
    ttl = "inf"


class MlViewerMonitorBinary(sdk2.Resource):
    """ Executable Monitor Binary for ML Viewer """
    any_arch = True            # Не зависит от архитектуры (что бы это ни значило)
    auto_backup = True         # Прикапываем на mds
    calc_md5 = True            # Создаем torrent id Skynet
    ttl = "inf"                # Храним вечно, чистим извне
    sync_upload_to_mds = True  # Пока не прикопаем - не считаем, что ресурс создался

    executable = True
    releasable = True
    releasers = RELEASERS

    arcadia_target = "ads/emily/viewer/backend/lib/monitor/bin/monitor"
    arcadia_revision = sdk2.parameters.String("Arcadia revision", default=None)


class MlViewerDocker(sdk2.Resource):
    """ Ml Viewer Docker Image Resource """
    releasable = True
