from __future__ import print_function

import json
import logging
import os
import shutil
import subprocess
import sys
import tempfile
import typing as tp

from sandbox.projects.ads.emily import resources

logger = logging.getLogger(__name__)

try:
    import builtins

    from future.utils import iteritems

    from library.python.find_root import detect_root

    from logos.libs.clients.sandbox.global_sandbox import SandboxClient
    from logos.libs.clients.sandbox.lib import BaseSandboxClient
    from logos.libs.clients.sandbox.local_sandbox import LocalSandboxClient
    from logos.libs.logging import configure_lib_loggers

    from ads.emily.storage.libs import errors
except ImportError:
    logging.warning("No external dependencies in build, try sdk2")

    import sandbox.sdk2 as sdk2

    configure_lib_loggers = None
    LocalSandboxClient = None
    builtins = None
    errors = None
    detect_root = None
    logger = logging

    def iteritems(obj, **kwargs):
        func = getattr(obj, "iteritems", None)
        if not func:
            func = obj.items
        return func(**kwargs)

    class SandboxClient(object):
        """ sdk2 mock """

        def __init__(self, token=None):
            self._resources = {}  # type: tp.Dict[int, sdk2.Resource]

        def resource_read(self, attrs, **kwargs):
            resource = resources.MlStorageCliBinary.find(**attrs).order(-sdk2.Resource.id).first()
            self._resources[resource.id] = resource
            return {"items": [{"id": resource.id}]}

        def download_resource(self, resource_id, temp_dir):
            if resource_id not in self._resources:
                self._resources[resource_id] = sdk2.Resource[resource_id]
            resource = self._resources[resource_id]
            subprocess.check_call(["sky", "get", "-wud", temp_dir, resource.skynet_id])


BINARY_CLIENT_NAME = "ml-storage-cli"
BINARY_CLIENT_DIR = "ads/emily/storage/models/client/bin"

# envs
ML_STORAGE_TOKEN = "ML_STORAGE_TOKEN"
ML_STORAGE_CLIENT_MODE = "ML_STORAGE_CLIENT_MODE"
ML_STORAGE_ENV = "ML_STORAGE_ENV"
ML_STORAGE_SANDBOX_TRANSPORT = "ML_STORAGE_SANDBOX_TRANSPORT"


def configure_logger(
    debug=False  # type: bool
):  # type: (...) -> None
    if configure_lib_loggers is not None:
        configure_lib_loggers({
            "__tests__": logging.DEBUG,
            "__main__": logging.DEBUG,
            "ads.emily": logging.INFO,
            "sandbox.projects.ads": logging.INFO,
            "sandbox.common.rest": logging.INFO,
        }, debug=debug)


def get_arcadia_root():
    # type: () -> tp.Optional[str]
    return detect_root("") if detect_root is not None else None


def get_token():
    # type: () -> tp.Optional[resources.EMlStorageClientMode]
    return os.getenv(ML_STORAGE_TOKEN)


def get_mode():
    # type: () -> tp.Optional[resources.EMlStorageClientMode]
    return os.getenv(ML_STORAGE_CLIENT_MODE)


def get_env():
    # type: () -> tp.Optional[resources.EMlStorageEnv]
    return os.getenv(ML_STORAGE_ENV)


def get_sandbox_transport():
    # type: () -> tp.Optional[resources.EMlStorageSandboxTransport]
    return os.getenv(ML_STORAGE_SANDBOX_TRANSPORT)


def get_arcadia_cli_path():
    # type: () -> str
    binary_path = os.path.join(BINARY_CLIENT_DIR, BINARY_CLIENT_NAME)
    root = get_arcadia_root()

    if root is None:
        logger.debug("Cli path: start search `{}` in test environment".format(binary_path))
        try:
            import yatest
            return yatest.common.binary_path(binary_path)
        except Exception:
            raise ValueError("Cli path: Can't extract from yatest")
    else:
        logger.debug("Cli path: start search `{}` in arcadia".format(binary_path))
        path = os.path.join(root, binary_path)
        if not os.path.exists(path):
            raise ValueError("Cli path: no such file found: `{}`".format(path))
        return path


class MlStorageBinaryClient(object):

    def __init__(
            self,                       # type: MlStorageBinaryClient
            mode=None,                  # type: tp.Optional[str]
            env=None,                   # type: tp.Optional[str]
            token=None,                 # type: tp.Optional[str]
            revision=None,              # type: tp.Optional[int]
            sb_client=None,             # type: tp.Optional[BaseSandboxClient]
            cli_path=None,              # type: tp.Optional[str]
    ):  # type: (...) -> None

        self._mode = resources.EMlStorageClientMode(get_mode() or mode or resources.EMlStorageClientMode.PROD.value)
        logger.info("Use {} mode".format(self._mode.value))

        self._env = resources.EMlStorageEnv(get_env() or env or resources.EMlStorageEnv.PROD.value)
        logger.info("Use {} env".format(self._env.value))

        transport = get_sandbox_transport()
        if transport:
            logger.info("Use {} transport".format(transport))

        self._token = token or get_token()

        if self._token is None and self._mode != resources.EMlStorageClientMode.TEST:
            raise ValueError("You should provide token in {} mode".format(self._mode.value))

        if LocalSandboxClient is None and self._mode == resources.EMlStorageClientMode.TEST:
            raise ValueError("Can't use TEST mode without logos deps")

        if sb_client is not None and cli_path is not None:
            raise AttributeError("Only one attribute required: sb_client or cli_path")

        self._sb_client = sb_client

        if self._mode == resources.EMlStorageClientMode.TEST:
            self._sb_client = self._sb_client or LocalSandboxClient()
        elif self._mode == resources.EMlStorageClientMode.PROD:
            self._sb_client = self._sb_client or SandboxClient(token=token)

        self.temp_dir = None
        if self._mode == resources.EMlStorageClientMode.PROD:
            self.temp_dir = tempfile.mkdtemp()
            self._client_bin = self._download_ml_storage_binary(sb_client=self._sb_client, temp_dir=self.temp_dir, revision=revision)
        else:
            cli_path = cli_path or get_arcadia_cli_path()
            self._client_bin = self._get_ml_storage_binary(cli_path)

    def __del__(self):
        if self.temp_dir is not None and os.path.exists(self.temp_dir):
            shutil.rmtree(self.temp_dir)

    def _get_client_command_env(self):
        # type: () -> tp.Dict[str, str]

        env = dict(os.environ)
        env_res_dict = {}
        for key, value in env.items():
            if key.startswith("ML_STORAGE") or key == "TMPDIR":
                env_res_dict[key] = value

        if self._token is not None:
            env_res_dict[ML_STORAGE_TOKEN] = str(self._token)

        if self._mode is not None:
            env_res_dict[ML_STORAGE_CLIENT_MODE] = self._mode.value
            if self._mode == resources.EMlStorageClientMode.TEST:
                env_res_dict["CACHE_DIR"] = tp.cast(LocalSandboxClient, self._sb_client).cache_dir

        if self._env is not None:
            env_res_dict[ML_STORAGE_ENV] = self._env.value

        return env_res_dict

    @staticmethod
    def _get_ml_storage_binary(
            path  # type: str
    ):  # type: (...) -> str
        logger.debug("Get {} binary from {}".format(BINARY_CLIENT_NAME, path))
        if not os.path.exists(path):
            raise ValueError("Cli path: No such file found: {}".format(path))
        return path

    @staticmethod
    def _download_ml_storage_binary(
            sb_client,      # type: BaseSandboxClient
            temp_dir,
            revision=None,  # type: tp.Optional[int]
    ):  # type: (...) -> str
        logger.debug("Get ML_STORAGE_CLI_BINARY {} from Sandbox".format(revision if revision else "latest released"))
        attrs = {"released": "stable"}
        if revision:
            attrs["arcadia_revision"] = revision
        resource_id = sb_client.resource_read(
            attrs=attrs,
            resource_type=str(resources.MlStorageCliBinary),
            limit=1,
        )["items"][0]["id"]

        binary_path = os.path.join(temp_dir, BINARY_CLIENT_NAME)
        logger.debug("Downloading: {} -> {}".format(resource_id, binary_path))
        sb_client.download_resource(resource_id, temp_dir)
        os.chmod(binary_path, 0o777)

        return binary_path

    def _run_client_command(
            self,    # type: MlStorageBinaryClient
            command  # type: tp.List[str]
    ):  # type: (...) -> str

        logger.info("Run: ./cli {}".format(" ".join(command)))
        # TODO: dim-gonch@ add live output
        process = subprocess.Popen([self._client_bin] + command,
                                   stdout=subprocess.PIPE, stderr=subprocess.PIPE,
                                   universal_newlines=True,
                                   env=self._get_client_command_env())
        stdout_str, stderr_str = process.communicate()
        sys.stderr.write(stderr_str)
        sys.stderr.write(stdout_str)
        sys.stderr.flush()

        if process.returncode != 0:
            if stdout_str == "":
                raise RuntimeError("Client command failed")
            try:
                exc_message = json.loads(stdout_str)
                exc_message_type = exc_message.get("type", "empty")
                logger.info("Error Type={}".format(exc_message_type))
                exc_type = getattr(errors, exc_message_type, getattr(builtins, exc_message_type, RuntimeError))
                message = exc_message.get("message", exc_message)
            except Exception:
                raise RuntimeError("Client command failed: {} with code {}".format(stdout_str, process.returncode))
            else:
                raise exc_type(message)

        return stdout_str

    def push(
            self,                       # type: MlStorageBinaryClient
            key,                        # type: str
            version,                    # type: str
            dumps,                      # type: tp.Dict[str, str]
            attrs=None,                 # type: tp.Optional[tp.Dict[str, str]]
            owner=None,                 # type: tp.Optional[str]
            untar=None,                 # type: tp.Optional[str]
            env=None,                   # type: tp.Optional[str]
            **kwargs                    # type: tp.Any
    ):  # type: (...) -> tp.Dict[str, tp.Any]
        """ Uploads model to storage"""
        args = [
            "push",
            "--key={}".format(key),
            "--version={}".format(version)
        ]
        for k, v in sorted(iteritems(dumps)):
            args += ["-D", "{}={}".format(k, v)]
        if attrs:
            for k, v in sorted(iteritems(attrs)):
                args += ["-A", "{}={}".format(k, v)]
        if owner is not None:
            args += ["--owner={}".format(owner)]
        if untar is not None:
            args += ["--untar={}".format(untar)]
        if env is not None:
            args += ["--env={}".format(env)]
        return json.loads(self._run_client_command(args))

    def list(
            self,         # type: MlStorageBinaryClient
            key,          # type: str
            prod=False,   # type: bool
    ):  # type: (...) -> tp.List[tp.Dict[str, tp.Any]]
        """ List models from storage by key """
        args = ["list", "--key", key]
        if prod:
            args.append("--prod")

        return json.loads(self._run_client_command(args))

    def info(
            self,                           # type: MlStorageBinaryClient
            key,                            # type: str
            version=None,                   # type: tp.Optional[str]
            latest=False,                   # type: bool
            prod=False,                     # type: bool
            draft=False,                    # type: bool
            do_not_load_dumps=False,   # type: bool
    ):  # type: (...) -> tp.Dict[str, tp.Any]
        """ Get model info from storage """
        args = ["info", "--key", key]
        if version:
            args.append("--version={}".format(version))
        if latest:
            args.append("--latest")
        if prod:
            args.append("--prod")
        if draft:
            args.append("--draft")
        if do_not_load_dumps:
            args.append("--do-not-load-dumps")
        return json.loads(self._run_client_command(args))

    def pull(
            self,               # type: MlStorageBinaryClient
            key,                # type: str
            version=None,       # type: str
            latest=False,       # type: bool
            prod=False,         # type: bool
            draft=False,        # type: bool
            dirpath=".",        # type: str
            dump_keys=None,     # type: tp.Optional[tp.Container[str]]
    ):  # type: (...) -> tp.Dict[str, tp.Any]
        """ Pull model from storage """
        dump_keys = dump_keys or []
        args = ["pull", "--key", key, "--dirpath", dirpath]
        if version:
            args.append("--version={}".format(version))
        if latest:
            args.append("--latest")
        if prod:
            args.append("--prod")
        if draft:
            args.append("--draft")
        for dump in dump_keys:
            args += ["-D", dump]
        return json.loads(self._run_client_command(args))

    def state(
            self,                       # type: MlStorageBinaryClient
            dump,                       # type: str
            prod=False,                 # type: str
            do_not_load_dumps=False,    # type: bool
    ):  # type: (...) -> tp.Dict[str, tp.Any]
        """ Dump storage state to disk

            Heavy command, use with batch command
        """
        args = ["state", "--dump", dump]
        if prod:
            args.append("--prod")
        if do_not_load_dumps:
            args.append("--do-not-load-dumps")
        return json.loads(self._run_client_command(args))

    def batch_info(
            self,                       # type: MlStorageBinaryClient
            models,                     # type: str
            state=None,                 # type: str
            prod=False,                 # type: bool
            do_not_load_dumps=False,    # type: bool
    ):  # type: (...) -> tp.Dict[str, tp.Any]
        """ Batch operation - Get model info from storage

        Heavy operation
        batch_info(models="models.json", state="state.json")

        return:
            {
                "models": [],
                "not_found_models": []
            }
        """
        args = ["batch"]
        if state:
            args.append("--state={}".format(state))
        if prod:  # prod fetch
            args.append("--prod")
        args += ["info", "--models", models]
        if prod:  # prod filter
            args.append("--prod")
        if do_not_load_dumps:
            args.append("--do-not-load-dumps")
        return json.loads(self._run_client_command(args))
