from __future__ import print_function

import logging
import os

import sandbox.common.errors as errors
import sandbox.projects.common.constants as consts
from sandbox import sdk2
from sandbox.common.types.resource import State
from sandbox.common.types.task import ReleaseStatus, Status
from sandbox.common.utils import chain
from sandbox.projects.ads.emily.utils import RESOURCES, get_all_resource_types, get_colored_text, get_resource_link, get_task_link
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.build.YaMake2 import YaMake2


class ReleaseStatusParameter(sdk2.parameters.String):
    default = "stable"
    choices = [(k, k) for k in ReleaseStatus]


class ResourceTypesParameter(sdk2.parameters.String):
    choices = [(k, k) for k in get_all_resource_types()]


class LogosBuildAndRelease(sdk2.Task):
    """
        YA_MAKE2 -> Resource -> Release
    """

    class Parameters(sdk2.Parameters):

        kill_timeout = 10 * 3600  # 10 hours
        description = "Logos Build and Release"

        with sdk2.parameters.Group("Resources") as resources:

            with sdk2.parameters.RadioGroup("Build or Select", per_line=2, description="To build resource YaMake2 will be used") as build_or_select:
                build_or_select.values["build"] = build_or_select.Value(value="Build new resource", default=True)
                build_or_select.values["select"] = build_or_select.Value(value="Select existing resource")

            with build_or_select.value["build"]:
                revision = sdk2.parameters.String(
                    "Arcadia branch / commit hash / revision",
                    description="Ex: r7312366, users/dim-gonch/logos, trunk, fd6aa52239c",
                    default="trunk",
                    required=True,
                )
                build_resource_type = ResourceTypesParameter("Resource type", description="Ex: ML_STORAGE_CLI_BINARY", required=True)
                fetch_arcadia_target = sdk2.parameters.Bool("Fetch binary path from `arcadia_target`", description="Get target from `arcadia_target` resource field", default=False)

                with fetch_arcadia_target.value[False]:
                    arcadia_target_path = sdk2.parameters.String("Arcadia target path", description="Path to target to build", required=True)

                set_resource_attrs = sdk2.parameters.Dict("Resource Attributes", description="Set resource attributes", required=False)

            with build_or_select.value["select"]:
                with sdk2.parameters.RadioGroup("Select resource", per_line=2, description="Get resource by id / Get resource by type") as select_resource:
                    select_resource.values["by_id"] = select_resource.Value(value="By resource id", default=True)
                    select_resource.values["by_type"] = select_resource.Value(value="By resource type")

                with select_resource.value["by_id"]:
                    resource_id = sdk2.parameters.Resource("Resource id", description="resource id", required=True)
                with select_resource.value["by_type"]:
                    resource_type = ResourceTypesParameter("Resource type", description="Ex: ML_STORAGE_CLI_BINARY", required=True)
                    resource_attrs = sdk2.parameters.Dict("Resource Attributes", description="Attributes to filter", required=False)

        with sdk2.parameters.Group("Release") as release:
            release_resource = sdk2.parameters.Bool("Release resource", default=True)
            release_status = ReleaseStatusParameter("Release status")
            update_resource_attrs = sdk2.parameters.Dict("Resource Attributes", description="Update resource attributes", required=False)

        with sdk2.parameters.Output:
            out_resource_id = sdk2.parameters.String("Resource id")
            released_resource = sdk2.parameters.Resource("Released Resource")

    def _ya_make(self, revision, path, resource_type, ttl="inf"):
        dir_path = os.path.dirname(path)
        task = self.server.task({
            "type": YaMake2.type,
            "description": "Build {} for {}\n".format(resource_type, self.id),
            "owner": self.owner,
            "notifications": self.server.task[self.id].read()["notifications"],
            "priority": {
                "class": self.Parameters.priority.cls,
                "subclass": self.Parameters.priority.scls,
            },
            "requirements": {
                "client_tags": "LINUX & ~INTEL_E5645",
            },
            "children": True,
            "custom_fields": [{"name": k, "value": v} for k, v in {
                "test": False,
                "result_single_file": True,
                "use_aapi_fuse": True,
                "use_arc_instead_of_aapi": False,
                "checkout_arcadia_from_url": "arcadia-arc:/#{}".format(revision),
                "targets": dir_path,
                "arts": path,
                "arts_source": "",
                "result_rt": resource_type,
                "result_rd": "Resource: {} for {}\n".format(resource_type, get_task_link(self.id)),
                "result_ttl": ttl,
                "build_system": consts.SEMI_DISTBUILD_BUILD_SYSTEM,
                "build_type": "release",
                "check_return_code": True,
                "strip_binaries": True,
                "ya_yt_store": True,
            }.items()],
        })
        task_id = task["id"]
        self.server.batch.tasks.start.update([task_id])
        self.Context.ya_make_task_id = task_id
        self._set_info("Building: {} | Task id: {}".format(resource_type, get_task_link(task_id)))
        raise sdk2.WaitTask([task_id], set(chain(Status.Group.FINISH, Status.Group.BREAK)), wait_all=True)

    def _set_info(self, status, color=None):
        if color:
            status = get_colored_text(status, color=color)
        self.set_info(status, do_escape=False)

    def _release_task(self, resource, release_status):
        task_id = resource.task_id
        task = None
        try:
            task = sdk2.Task[task_id]
        except errors.TaskError as e:
            logging.error("{}".format(e))
            self._set_info("Release by resource task id", color="forestgreen")
        if task and task.status != "RELEASED" or getattr(resource, "released", None) != release_status:
            eh.ensure(not self.Context.releasing, "Releasing failed")
            self.server.release(
                task_id=task_id,
                type=release_status,
                subject=self.Parameters.description,
            )
            self.Context.releasing = True
            self._set_info("Releasing: {} with status: {}".format(get_task_link(task_id), release_status))
            raise sdk2.WaitTask([task_id], [Status.RELEASED], wait_all=True, timeout=60)

    def _get_resource(self, resource_type, task_id=None, **attributes):
        find_args = {
            "state": State.READY,
            "attrs": attributes,
        }
        if task_id is not None:
            find_args["task_id"] = task_id

        resource = sdk2.Resource[resource_type].find(**find_args).first()
        if resource is None:
            raise ValueError("No ready resources '{}'{} with attrs {} found".format(resource_type, " for task {}".format(task_id) if task_id else "", attributes))
        return sdk2.Resource[resource.id]

    def _fetch_arcadia_target(self, resource_type):
        for resource in RESOURCES:
            if resource_type == str(resource):
                return resource.arcadia_target
        raise ValueError("Can't fetch arcadia_target from resource: {}".format(resource))

    def on_execute(self):
        if not self.Context.resource_id:
            if self.Parameters.build_or_select == "build":
                if not self.Context.ya_make_task_id:
                    if self.Parameters.fetch_arcadia_target:
                        arcadia_path = self._fetch_arcadia_target(self.Parameters.build_resource_type)
                    else:
                        arcadia_path = self.Parameters.arcadia_target_path
                    if arcadia_path is None:
                        raise ValueError("Arcadia path is not set")
                    self._ya_make(self.Parameters.revision, arcadia_path, self.Parameters.build_resource_type)

                resource = self._get_resource(
                    resource_type=self.Parameters.build_resource_type,
                    task_id=self.Context.ya_make_task_id,
                )
                for key, value in self.Parameters.set_resource_attrs.items():
                    setattr(resource, key, value)
                self._set_info("Built resource: {} | id: {}".format(resource.type, get_resource_link(resource.id)))

            elif self.Parameters.select_resource == "by_id":
                resource = sdk2.Resource[self.Parameters.resource_id]
                self._set_info("Get resource: {} | by id: {}".format(resource.type, get_resource_link(resource.id)))
            else:
                resource = self._get_resource(
                    resource_type=self.Parameters.resource_type,
                    **self.Parameters.resource_attrs
                )
                self._set_info("Found resource: {} | id: {}".format(resource.type, get_resource_link(resource.id)))

            self.Context.resource_id = resource.id
        else:
            resource = sdk2.Resource[self.Context.resource_id]

        if self.Parameters.release_resource:
            self._release_task(resource, self.Parameters.release_status)
            self._set_info("Released with status: {}".format(self.Parameters.release_status))
            self.Parameters.released_resource = resource

        for key, value in self.Parameters.update_resource_attrs.items():
            setattr(resource, key, value)

        self.Parameters.out_resource_id = resource.id
