import json
import logging
import time
from typing import List, Dict, Tuple

from library.python import resource
from sandbox import sdk2
import sandbox.common.types.task as ctt
import yt.yson as yson
from yt.yson import YsonList

import sandbox.projects.ads.emily.tasks.export_lm_scheme_and_config.lib.yt_export as yt_export
from sandbox.projects.ads.emily.tasks.export_lm_scheme_and_config.lib.py23.modes import Modes
from sandbox.projects.ads.emily.tasks.export_lm_scheme_and_config.lib.py23.sandbox_parameters import Parameters as ExportParameters  # noqa: E501


class ExportLinearModelTablesToYT(sdk2.Task):
    class Requirements(sdk2.Requirements):
        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='lm_exporting_to_yt', weight=1, capacity=1)
            ]
        )
        cores = 1
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(ExportParameters):  # pylint: disable=too-few-public-methods # noqa
        pass

    def on_execute(self):
        TABLES = (  # name, schema file key, content resource
            (yt_export.TableNames.LM_CONFIG, "LinearModelConfig-schema.yson", self.Parameters.lm_config_content),
            (yt_export.TableNames.LM_SCHEME, 'LinearModelScheme-schema.yson', self.Parameters.lm_scheme_content),
        )
        tables_by_name: List[Tuple[str, YsonList, List[Dict]]] = []
        yt_token = self.Parameters.yt_token.data().get(self.Parameters.yt_token.default_key, None) or \
            self.Parameters.yt_token.data().get("yt_token", None) or self.Parameters.yt_token.data()["token"]

        for table_name, resource_name, content_resource in TABLES:
            schema_resource_data = resource.find(resource_name)
            schema = yson.loads(schema_resource_data)
            content = json.loads(sdk2.ResourceData(content_resource).path.read_bytes())
            tables_by_name.append((table_name, schema, content))
        should_create_new_snapshot = self.Parameters.mode in (Modes.ONLY_CREATE_SNAPSHOT, Modes.FULL_MODE)
        should_change_links = self.Parameters.mode in (Modes.ONLY_CHANGE_LINKS, Modes.FULL_MODE)

        if abs(self.Parameters.extremely_dangerous_disable_snapshot_verification - time.time()) < 1000:
            logging.warning("REALLY DANGEROUS! UNIX time entered correctly, snapshot verification is disabled")
            yt_export.EXTREMELY_DANGEROUS_DISABLE_SNAPSHOT_VERIFICATION = True
        yt_export.main(
            tables_by_name,
            clusters=self.Parameters.clusters,
            yt_dir_path=self.Parameters.yt_dir_path,
            yt_token=yt_token,
            snapshot_id=self.Parameters.snapshot_id,
            should_create_new_snapshot=should_create_new_snapshot,
            should_change_links=should_change_links,
            sandbox_task=self
        )
