import logging

import simplejson as json

import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.ads.eshow.common.utils import get_binary_path, get_task_by_id

from sandbox.projects.ads.eshow.resources import AdsReachZcCalculatorBinaryV2
from sandbox.projects.ads.eshow.calculate_reach_zc import AdsCalculateReachZc
from sandbox.projects.ads.eshow.calculate_reach_zc.lib.constants import (
    COMMERCE_ALL, COMMERCE_ONLY_COMMERCE, COMMERCE_ONLY_NOT_COMMERCE,
    MODE_DO_NOTHING
)
from . import utils


class AdsAutoStatsUploader(sdk2.Task):
    """Task to upload metrics & stats to StarTrek Ticket"""

    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment(
                package_name="startrek-client", version="2.5",
                custom_parameters=["requests==2.18.4", "--upgrade-strategy only-if-needed"]
            ),
        )

    class Parameters(sdk2.Task.Parameters):
        description = "AutoStatsUploader default description"
        tags = ["ads", "reach-product"]
        kill_timeout = 12 * 60 * 60  # 12h
        max_restarts = 5

        startrek_secret_name = sdk2.parameters.YavSecret(
            label="StarTrek token",
            description="Secret to StarTrek API in YAV",
            required=True
        )
        conf = sdk2.parameters.String(
            label="Config",
            description="Example: https://paste.yandex-team.ru/4612501/text",
            default="https://paste.yandex-team.ru/4612489/text",
            required=True
        )
        auto_stop = sdk2.parameters.Bool(
            label="Auto stop",
            description="If active, metrics are not further posted at the end of the experiment",
            default=True
        )
        no_comment = sdk2.parameters.Bool(
            label="No Comment",
            description="If active, metrics will not be published, see the result in \"Output parameters\"",
            default=False
        )
        update_comment = sdk2.parameters.Bool(
            label="Update comment",
            description="If not active, create a new comment every launch",
            default=True
        )
        comment_header = sdk2.parameters.String(
            label="Comment header",
            description="[Optional] The header of the comment"
        )

        with sdk2.parameters.Group("Binary Resource") as resource_params:
            i_am_pro = sdk2.parameters.Bool(
                label="Change default binary resource",
                default=False
            )
            with i_am_pro.value["true"]:
                exp_stats_search_config = sdk2.parameters.String(
                    label="Exp Stats",
                    description="Json with parameters of searching binary",
                    default="""{"type": "ADS_EXP_STATS_BINARY", "attrs": "{\\"released\\": \\"stable\\"}"}""",
                )
                reach_zc_search_config = sdk2.parameters.String(
                    label="Reach zC",
                    description="Json with parameters of searching binary",
                    default="""{"type": "ADS_REACH_ZC_CALCULATOR_BINARY", "attrs": "{\\"released\\": \\"stable\\"}"}""",
                )
                cpc_zc_search_config = sdk2.parameters.String(
                    label="CPC zC",
                    description="Json with parameters of searching binary",
                    default="""{"type": "ADS_CPC_ZC_BINARY", "attrs": "{\\"released\\": \\"stable\\"}"}""",
                )
                apc_check_search_config = sdk2.parameters.String(
                    label="APC Check",
                    description="Json with parameters of searching binary",
                    default="""{"type": "ADS_APC_CHECK_BINARY", "attrs": "{\\"released\\": \\"stable\\"}"}""",
                )
                reach_zc_calculator_v2_binary = sdk2.parameters.Resource(
                    label="Reach zC v2",
                    description="If stay empty - will find latest released",
                    resource_type=AdsReachZcCalculatorBinaryV2,
                    state=(ctr.State.READY, )
                )
            cpc_zc_conf = sdk2.parameters.String(
                label="CPC zC config",
                default="https://paste.yandex-team.ru/4858414/text",
            )

        with sdk2.parameters.Group("Conf Parameters") as conf_params:
            ticket = sdk2.parameters.String(
                label="StarTrek Ticket",
                description="Metrics will be posted to this ticket",
                required=True
            )

            with sdk2.parameters.String(
                label="Prefix of TestIDs",
                description="ab (ActiveTestIds), pcode (PcodeTestIds), yaexp (YandexExpBoxes)",
                default="ab",
                required=True
            ) as testids_prefix:
                testids_prefix.values.ab = "ab"
                testids_prefix.values.pcode = "pcode"
                testids_prefix.values.yaexp = "yaexp"

            testids = sdk2.parameters.List(
                label="TestIDs",
                description="List[str] or comma-separated values"
            )
            etalon = sdk2.parameters.String(
                label="Etalon TestID"
            )
            start_date = sdk2.parameters.String(
                label="Start date",
                description="In the YYYYMMDD format (if not set current data will be used)"
            )
            end_date = sdk2.parameters.String(
                label="End date",
                description="In the YYYYMMDD format (if not set current data will be used)"
            )

        with sdk2.parameters.Group("zC V2") as zc_v2_params:
            zc_v2_common_params = sdk2.parameters.String(
                label="Common parameters",
                description="Do not put experiments and date here, they are provided elsewhere",
                default="--keys OrderID --full-traffic-zc --commerce"
            )
            zc_v2_override_params = sdk2.parameters.Dict(
                label="Override configs",
                description="There parameters append to common (name collision is not permitted!)"
            )

        with sdk2.parameters.Group("YT Parameters") as yt_params:
            yt_secret_name = sdk2.parameters.YavSecret(
                label="YT token",
                description="Secret to YT_TOKEN in YAV",
                required=True
            )
            yt_proxy = sdk2.parameters.String(
                label="YT cluster",
                description="export YT_PROXY",
                default="hahn"
            )
            yt_pool = sdk2.parameters.String(
                label="YT Pool",
                description="export YT_POOL"
            )

        with sdk2.parameters.Output():
            result = sdk2.parameters.String(label="Task execution wiki-result")

    @sdk2.header()
    def header(self):
        if self.Parameters.ticket and not self.Parameters.no_comment:
            return "<a href=\"https://st.yandex-team.ru/{ticket}\">{ticket}</a>".format(
                ticket=self.Parameters.ticket
            )

    def get_default_wiki_comment_header(self):
        return "((https://st.yandex-team.ru/BSDEV-83315#60bf62b22eb3cd7015f93b05 {type}))".format(
            type=self.type
        )

    def on_create(self):
        """ Called on task creation """
        # hardcode :)
        if self.author == "berezniker":
            self.Parameters.startrek_secret_name = sdk2.yav.Secret(
                secret="sec-01f6pd296enwmd6x9kcfjfzy1g",
                default_key="nirvana-secret"
            )
            self.Parameters.yt_secret_name = sdk2.yav.Secret(
                secret="sec-01ettgxzpb5c0sdmemwfg78wev",
                default_key="nirvana-secret"
            )

    def on_save(self):
        """ Called when updating task in status DRAFT """
        if not self.Parameters.yt_pool:
            self.Parameters.yt_pool = self.author

        if not self.Parameters.testids:
            testids = utils.get_testids_from_ticket(self.Parameters.ticket)
            self.Parameters.testids = utils.add_prefix_to_testids(testids, self.Parameters.testids_prefix)

        if not self.Parameters.etalon and self.Parameters.testids:
            self.Parameters.etalon = min(self.Parameters.testids).split(',')[0]

    def on_execute(self):
        """ Called when task is executing on the host """
        with self.memoize_stage.running_subtasks():
            conf = utils.load_config(self.Parameters.conf, parse_yaml=True)
            utils.validate(args=self.Parameters, conf=conf)
            self.Context.conf = conf
            if self.Parameters.auto_stop:
                if utils.check_auto_stop(
                    left_date=conf["params"]["end_date"],
                    ticket=self.Parameters.ticket
                ):
                    return

            self.run_zc_v2_configs()
            result = utils.process(conf=conf)
            self.Context.result = result

        with self.memoize_stage.waiting_for_zc(2):
            zc_v2_result = self.wait_for_zc_v2_subtasks()

        result_wiki = utils.to_wiki_format(
            conf=self.Context.conf,
            result=utils.merge_results(self.Context.result, zc_v2_result),
            task_id=self.id,
            header=self.Parameters.comment_header or self.get_default_wiki_comment_header()
        )
        self.Parameters.result = result_wiki

        if not self.Parameters.no_comment:
            utils.upload_to_startrek(
                args=self.Parameters,
                result_wiki=result_wiki,
                comment_header=self.Parameters.comment_header or self.get_default_wiki_comment_header()
            )

    def run_zc_v2_configs(self):
        logging.info("Spawning zC v2 subtasks")
        binary_path = get_binary_path(self.Parameters.reach_zc_calculator_v2_binary, AdsReachZcCalculatorBinaryV2)
        subtasks = dict()

        for (key, override_params) in self.Parameters.zc_v2_override_params.items():
            subtask, args = self._run_zc_v2_task(binary_path, key, self.Parameters.zc_v2_common_params, override_params)
            subtask.enqueue()
            subtasks[key] = subtask

        self.Context.zc_v2_subtasks = {key: subtask.id for (key, subtask) in subtasks.items()}

    def wait_for_zc_v2_subtasks(self):
        wait_for = dict()
        zc_v2_result = dict()

        for (key, subtask_id) in self.Context.zc_v2_subtasks.items():
            subtask = get_task_by_id(subtask_id)

            if subtask.status == ctt.Status.SUCCESS:
                logging.info("zC v2 subtask for key {} successfully finished".format(key))
                zc_v2_result[key] = (subtask.id, subtask.Parameters.zc_result_wiki)
            elif subtask.status in ctt.Status.Group.BREAK or subtask.status in (ctt.Status.FAILURE, ctt.Status.DELETED):
                logging.info("zC v2 subtask for key {} failed".format(key))
                zc_v2_result[key] = (subtask.id, None)
            else:
                wait_for[key] = subtask

        if wait_for:
            logging.info("Waiting for following zC v2 subtasks: {}".format(
                ", ".join("{} ({})".format(key, subtask.id) for (key, subtask) in wait_for.items())
            ))
            raise sdk2.WaitTask(wait_for.values(), (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK), wait_all=True)
        else:
            return zc_v2_result

    def _run_zc_v2_task(self, binary, key, def_param_string, override_param_string):
        logging.info("Starting zC v2 subtask for key {}".format(key))
        cmd = [binary, "--exec-mode", MODE_DO_NOTHING]
        fname = "zc_v2_args_{}.json".format(key)
        cmd.extend(("--parse-args-to-json", fname))
        cmd.extend(def_param_string.split(" "))
        cmd.extend(override_param_string.split(" "))

        cmd.extend(("--range", "{start_date}..{end_date}".format(
            start_date=self.Context.conf["params"]["start_date"],
            end_date=self.Context.conf["params"]["end_date"])
        ))
        cmd.extend((["--experiments"] + self.Parameters.testids))

        if self.Parameters.etalon:
            cmd.extend(("--etalon", self.Parameters.etalon))

        if self.Parameters.yt_proxy:
            cmd.extend(("--yt-proxy", self.Parameters.yt_proxy))

        if self.Parameters.yt_pool:
            cmd.extend(("--yt-pool", self.Parameters.yt_pool))

        with sdk2.helpers.ProcessLog(self, logger="parse_args_zc_v2_{}".format(key)) as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr)

        with open(fname, "r") as fh:
            args = json.load(fh)

        logging.info("Parsed arguments for subtask: {}".format(", ".join("{} = {}".format(k, v) for (k, v) in args.items())))
        subtask = AdsCalculateReachZc(
            self,
            description="Subtask of {}, zc calculation for key \"{}\"".format(self.id, key),
            secret_name=self.Parameters.yt_secret_name,
            date_range=args["range"],
            experiments=args["experiments"], etalon=args.get("etalon", None),
            product_types=args["product_types"],
            full_traffic_zc=args.get("full_traffic_zc", None),
            keys=args.get("keys", None), max_groups=args.get("max_groups", None),
            conditions=args.get("conditions", None), not_conditions=args.get("not_conditions", None),
            actions=args["action_coefficients"],
            zc_method=args.get("method", None),
            yt_pool=args.get("yt_pool", None), yt_proxy=args.get("yt_proxy", None),
            yt_data_weight=args.get("yt_data_weight", None),
            commerce=COMMERCE_ONLY_COMMERCE if args.get("commerce", False) else (
                COMMERCE_ONLY_NOT_COMMERCE if args.get("not_commerce", False) else COMMERCE_ALL
            )
        )
        subtask.save()
        logging.info("Subtask ID: {}".format(subtask.id))
        return subtask, args
