from datetime import datetime, timedelta
from itertools import chain
import requests
import logging
import json
import yaml
import copy
import os

from .constants import (
    AB_API_TASK_URL, DATE_FORMAT, DATE_FORMAT_, DATE_FORMAT_HOUR_MINUTES,
    DEFAULT_YT_PROXY, GET_TESTIDS_FIELD, PASTE_YANDEX_TEAM_URL,
    STARTREK_BASE_URL, STARTREK_USERAGENT, WIKI_HEADER, WIKI_TEMPLATE
)

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess


def get_today_date():
    return datetime.now().date().strftime(DATE_FORMAT)


def get_shifted_date(offset):
    return (datetime.now() + timedelta(days=offset)).strftime(DATE_FORMAT)


def get_token_by_name(secret_name):
    return secret_name.data()[secret_name.default_key]


def check_auto_stop(left_date, ticket):
    logging.info("Check Auto Stop")
    running_date_end = get_date_end_from_ticket(ticket)
    if running_date_end:
        date_end = datetime.strptime(running_date_end, DATE_FORMAT_)
        left_date = datetime.strptime(left_date, DATE_FORMAT)
        if date_end < left_date:
            logging.info("Auto Stop: {} < {}".format(date_end, left_date))
            return True
    return False


def get_date_end_from_ticket(ticket):
    logging.info("Get running date end from ticket: {ticket}".format(ticket=ticket))
    if ticket.startswith("EXPERIMENTS-"):
        response = requests.get(AB_API_TASK_URL + ticket)
        return json.loads(response.text)["running_date_end"]
    else:
        logging.info("Incorrect ticket name: {ticket}".format(ticket=ticket))
    return


def get_testids_from_ticket(ticket):
    testids = list()
    logging.info("Get TestIDs from ticket: {ticket}".format(ticket=ticket))
    if ticket.startswith("EXPERIMENTS-"):
        response = requests.get(AB_API_TASK_URL + ticket)
        testids = json.loads(response.text)["testids"]
    else:
        logging.info("Incorrect ticket name: {ticket}".format(ticket=ticket))
    return testids


def add_prefix_to_testid(testid, prefix):
    if isinstance(testid, int) or testid.isdigit():
        testid_with_prefix = "{prefix}{testid}".format(testid=testid, prefix=prefix)
        logging.warning("Change testid: {testid} -> {testid_with_prefix}".format(
            testid=testid,
            testid_with_prefix=testid_with_prefix
        ))
        return testid_with_prefix
    return testid


def add_prefix_to_testids(testids, prefix):
    return list(map(lambda t: add_prefix_to_testid(t, prefix), testids))


def validate_testids(args_testids, prefix):
    testids = list()
    for testid in args_testids:
        for tid in testid.split(','):
            testids.append(add_prefix_to_testid(tid, prefix))
    return testids


def load_config(path_to_config, parse_yaml=False):
    if not path_to_config.startswith(PASTE_YANDEX_TEAM_URL):
        raise ValueError("Incorrect path to config: {}".format(path_to_config))

    if not path_to_config.endswith("text"):
        logging.warning("Add \"/text\" to path: {}".format(path_to_config))
        path_to_config = path_to_config + "/text"

    logging.info("Load: {}".format(path_to_config))
    config = requests.get(url=path_to_config).text

    if parse_yaml:
        logging.debug("Parse YAML")
        return yaml.safe_load(config)

    return config


def load_cpc_zc_config(path_to_conf, params):
    conf = load_config(path_to_conf)
    conf = conf.format(**params)
    logging.debug("CPC zC Config:\n" + conf)

    path_to_local_conf = "zc.conf"
    with open(path_to_local_conf, mode='w') as f:
        f.write(conf)

    return path_to_local_conf


def validate_date_format(date):
    try:
        datetime.strptime(date, DATE_FORMAT)
    except ValueError:
        raise ValueError("Incorrect data format, should be YYYYMMDD.")


def update_conf(args, conf):
    if not conf.get("params"):
        logging.debug("Task params not specified in config.")
        conf["params"] = dict()

    conf["params"]["testids_field"] = GET_TESTIDS_FIELD[args.testids_prefix]

    if args.testids:
        conf["params"]["testids"] = validate_testids(args.testids, args.testids_prefix)
    else:
        logging.info("TestIDs not specified.")
        testids = get_testids_from_ticket(args.ticket)
        conf["params"]["testids"] = add_prefix_to_testids(testids, args.testids_prefix)
    conf["params"]["testids"] = ','.join(conf["params"]["testids"])
    assert conf["params"]["testids"], "TestIDs not specified."
    logging.debug("TestIDs: {}".format(conf["params"]["testids"]))

    conf["params"]["etalon"] = args.etalon or min(conf["params"]["testids"])
    conf["params"]["etalon"] = add_prefix_to_testid(conf["params"]["etalon"], args.testids_prefix)
    logging.debug("Etalon TestID: {}".format(conf["params"]["etalon"]))

    conf["params"]["start_date"] = args.start_date or conf["params"].get("start_date") or get_today_date()
    if int(conf["params"]["start_date"]) < 0:
        conf["params"]["start_date"] = get_shifted_date(int(conf["params"]["start_date"]))
    validate_date_format(conf["params"]["start_date"])

    conf["params"]["end_date"] = args.end_date or conf["params"].get("end_date") or get_today_date()
    if int(conf["params"]["end_date"]) < 0:
        conf["params"]["end_date"] = get_shifted_date(int(conf["params"]["end_date"]))
    validate_date_format(conf["params"]["end_date"])

    logging.info("Date range {start_date}..{end_date}".format(
        start_date=conf["params"]["start_date"],
        end_date=conf["params"]["end_date"]
    ))

    conf["yt_token"] = get_token_by_name(args.yt_secret_name)
    conf["yt_proxy"] = args.yt_proxy or DEFAULT_YT_PROXY
    conf["yt_pool"] = args.yt_pool or args.author

    return conf


def search_config(json_config):
    params = json.loads(json_config)
    logging.debug("params: {}".format(params))
    resource = sdk2.Resource.find(**params).order(-sdk2.Resource.id).first()
    return str(sdk2.ResourceData(resource).path)


def validate(args, conf):
    logging.info("Validate arguments & config")
    conf = update_conf(args, conf)

    if conf.get("exp_stats"):
        logging.info("Upload ExpStats binary")
        if not args.exp_stats_search_config:
            logging.error("ExpStats binary not specified.")
            raise AttributeError("ExpStats binary not specified.")
        conf["params"]["exp_stats"] = search_config(args.exp_stats_search_config)

    if conf.get("reach_zc"):
        logging.info("Upload Reach zC binary")
        if not args.reach_zc_search_config:
            logging.error("Reach zC binary not specified.")
            raise AttributeError("Reach zC binary not specified.")
        conf["params"]["reach_zc"] = search_config(args.reach_zc_search_config)
        logging.info("Upload APC Check binary")
        if not args.apc_check_search_config:
            logging.error("APC Check binary not specified.")
            raise AttributeError("APC Check binary not specified.")
        conf["params"]["apc_check_path"] = search_config(args.apc_check_search_config)

    if conf.get("cpc_zc"):
        logging.info("Upload CPC zC binary")
        if not args.cpc_zc_conf:
            logging.error("CPC zC config not specified.")
            raise AttributeError("CPC zC config not specified.")
        conf["params"]["cpc_zc_conf_paste"] = args.cpc_zc_conf
        conf["params"]["cpc_zc_conf"] = load_cpc_zc_config(args.cpc_zc_conf, conf["params"])
        if not args.cpc_zc_search_config:
            logging.error("CPC zC binary not specified.")
            raise AttributeError("CPC zC binary not specified.")
        conf["params"]["cpc_zc"] = search_config(args.cpc_zc_search_config)

    logging.debug("Configuration:\n{}".format(conf))

    try:
        from startrek_client import Startrek
        logging.info("Test Connection to Startrek API")
        client = Startrek(
            useragent=STARTREK_USERAGENT,
            base_url=STARTREK_BASE_URL,
            token=get_token_by_name(args.startrek_secret_name)
        )
        issue = client.issues[args.ticket]
        logging.info("Metrics will be posted to {ticket} by {login}".format(
            ticket=args.ticket,
            login=issue.createdBy.login
        ))
    except BaseException as e:
        logging.error(e)
        raise e

    return


def bytes_to_str(text):
    if isinstance(text, bytes):
        text = text.decode("utf-8")
    return str(text)


def pretty_command(query, params):
    params_copy = params.copy()
    params_copy.update({
        "exp_stats": "./exp_stats",
        "reach_zc": "./reach_zc_calculator",
        "cpc_zc": "./zc",
        "apc_check_path": "apc_check"
    })
    return query.format(**params_copy)


def process(conf):
    env = os.environ.copy()
    env["YT_TOKEN"] = conf["yt_token"]
    env["YT_PROXY"] = conf["yt_proxy"]
    env["YT_POOL"] = conf["yt_pool"]
    processes, result = list(), list()

    # Run all commands in subprocesses
    for command in chain(conf.get("exp_stats", []), conf.get("reach_zc", []), conf.get("cpc_zc", [])):
        cmd = command["query"].format(**conf.get("params", dict()))
        logging.info("Run command: {}".format(cmd))
        proc = subprocess.Popen(
            args=cmd.split(), env=env,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )
        processes.append((
            command.get("title", "NoTitle"),
            pretty_command(query=command["query"], params=conf.get("params", dict())),
            proc
        ))

    # Wait for the result
    success_counter = 0
    for title, cmd, proc in processes:
        out, err = proc.communicate()
        if err:
            logging.warning("[stderr] {title}:\n{message}".format(
                title=title,
                message=bytes_to_str(err)
            ))
        if out:
            logging.info("[stdout] {title}:\n{message}".format(
                title=title,
                message=bytes_to_str(out)
            ))
            success_counter += 1
            result.append((title, cmd, bytes_to_str(out).strip()))

    assert success_counter, "All launches FAILED! See common.log / debug.log for more information."

    return result


def to_wiki_format(conf, result, task_id, header=""):
    wiki_header = WIKI_HEADER.format(
        header=header,
        start_date=conf["params"]["start_date"],
        end_date=conf["params"]["end_date"],
        sandbox_task="((https://sandbox.yandex-team.ru/task/{}/view Sandbox Task))\n".format(task_id)
    )
    wiki_body = ''.join([
        WIKI_TEMPLATE.format(title=title, command=command, table=table)
        for title, command, table in result
    ])
    return (wiki_header + wiki_body).strip()


def merge_results(common_result, zc_v2_result):
    result = copy.deepcopy(common_result)

    for (key, (subtask_id, zc_value)) in sorted(zc_v2_result.items()):
        cmd = "https://sandbox.yandex-team.ru/task/{}".format(subtask_id)
        result.append((key, cmd, zc_value or "**Calculation failed**"))

    return result


def upload_to_startrek(args, result_wiki, comment_header):
    logging.info("Add comment to ticket: {ticket}:\n{comment}".format(
        ticket=args.ticket,
        comment=result_wiki
    ))

    logging.info("Connecting to Startrek API")
    from startrek_client import Startrek
    client = Startrek(
        useragent=STARTREK_USERAGENT,
        base_url=STARTREK_BASE_URL,
        token=get_token_by_name(args.startrek_secret_name)
    )
    issue = client.issues[args.ticket]
    if args.update_comment:
        for comment in issue.comments.get_all():
            if comment_header in comment.text:
                logging.info("Comment will be updated")
                result_wiki += "\n++!!(gray)Updated on {}!!++".format(datetime.now().strftime(DATE_FORMAT_HOUR_MINUTES))
                comment.update(text=result_wiki)
                return
        else:
            logging.info("Comment for the update was not found")

    logging.info("Comment will be create")
    issue.comments.create(text=result_wiki)
    return
