import json
import logging

from sandbox import sdk2
import sandbox.common.types.resource as ctr
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.ads.eshow.resources import AdsReachZcCalculatorBinaryV2
from sandbox.projects.ads.eshow.common.utils import get_binary_path
from sandbox.projects.ads.eshow.calculate_reach_zc.lib.constants import MODE_CALC_USER_BUCKET_STATS

TMP_JSON = "exp_stats.json"


class AdsReachZcCalculateLimitedExpStats(sdk2.Task):
    """Master-task to calculate reach zC"""

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1 << 10
        ram = 2 << 10

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 2 * 60 * 60

        binary_resource = sdk2.parameters.Resource(
            "Reach zC calculator binary",
            description="Sandbox resource with compiled binary. If empty - will find latest released",
            resource_type=AdsReachZcCalculatorBinaryV2,
            state=(ctr.State.READY,)
        )
        yt_directory = sdk2.parameters.String(
            "YT folder with premapped tables",
            required=True
        )
        secret_name = sdk2.parameters.YavSecret(
            "Secret in YAV with access to YT. Use #key for key",
            required=True
        )

        with sdk2.parameters.Group("Execution Parameters") as exec_params:
            yt_proxy = sdk2.parameters.String(
                "YT proxy",
                required=True, default="hahn"
            )
            yt_pool = sdk2.parameters.String(
                "YT pool",
                required=True, default="ads-research"
            )

        with sdk2.parameters.Output():
            exp_stats = sdk2.parameters.String("Experiments stats")

    def on_execute(self):
        logging.info("Starting tables preparation stage")
        binary_path = get_binary_path(self.Parameters.binary_resource, AdsReachZcCalculatorBinaryV2)

        cmd = self._get_arguments(binary_path=binary_path)
        env_vars = {
            "YT_PROXY": self.Parameters.yt_proxy,
            "YT_TOKEN": self.Parameters.secret_name.data()[self.Parameters.secret_name.default_key]
        }
        logging.info("Command to run: {}".format(" ".join(cmd)))

        with sdk2.helpers.ProcessLog(self, logger="zc_calculator") as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr, env=env_vars)

        logging.info("Calculation finished, collecting data")

        with open(TMP_JSON, "r") as fh:
            exp_stats = json.load(fh)

        self.Parameters.exp_stats = json.dumps(exp_stats)

    def _get_arguments(self, binary_path):
        return [
            binary_path,
            "--exec-mode", MODE_CALC_USER_BUCKET_STATS,
            "--premapped-tables-path", self.Parameters.yt_directory,
            "--dump-exp-stats-to", TMP_JSON,
            "--yt-pool", self.Parameters.yt_pool,
            "--yt-proxy", self.Parameters.yt_proxy,
        ]
