import itertools
import json
import logging
import math
import re

from sandbox import sdk2
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk import environments as envs
from sandbox.common import errors
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.ads.eshow.common.utils import (
    get_binary_path, get_task_by_id, get_yt_client, get_yt_table_attr, make_random_yt_dir
)

from sandbox.projects.ads.eshow.resources import AdsReachZcCalculatorBinaryV2
from sandbox.projects.ads.eshow.calculate_reach_zc.lib.constants import (
    HTML_STYLE,
    CONTROL_EXP_ID, ZC_APC_METHODS,
    COMMERCE_ALL, COMMERCE_ONLY_COMMERCE, COMMERCE_ONLY_NOT_COMMERCE,
    EXP_PREFIXES,
    MODE_PREMAP,
    ATTR_EXPID, ATTR_ACTION, ATTR_ACTION_VIEW, ATTR_IS_CONTROL
)
from sandbox.projects.ads.eshow.calculate_reach_zc.lib.output import print_zc
from sandbox.projects.ads.eshow.calculate_reach_zc.calculate_exp_stats import AdsReachZcCalculateLimitedExpStats
from sandbox.projects.ads.eshow.calculate_reach_zc.read_table_from_yt import AdsReachZcReadPremappedTableFromYt
from sandbox.projects.ads.eshow.calculate_reach_zc.run_apc_check import AdsRunApcCheckOnPreparedTable

import utils

PRODUCT_TYPES = ["media-creative-reach", "video-creative-reach", "video-creative-reach-non-skippable"]
DT_REGEXPR = re.compile(r"\d{8}(\d{2})?(\.\.(\d{8}(\d{2})?)?)?")
RE_EXP = re.compile(r"({})\d{{6,}}".format("|".join(sorted(EXP_PREFIXES.keys()))))
DEFAULT_WORKING_DIR = "//tmp/reach_zc_calculation"


class AdsCalculateReachZc(sdk2.Task):
    """Master-task to calculate reach zC"""

    class Requirements(sdk2.Task.Requirements):
        disk_space = 2 << 10
        ram = 1 << 10
        environments = [envs.PipEnvironment("yandex-yt")]
        tags = ["ads", "zC"]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 24 * 60 * 60
        description = "Reach products zC calculation"

        binary_resource = sdk2.parameters.Resource(
            "Reach zC calculator binary",
            description="Sandbox resource with compiled binary. If empty - will find latest released",
            resource_type=AdsReachZcCalculatorBinaryV2,
            state=(ctr.State.READY, ),
            # default=2442391243
        )
        secret_name = sdk2.parameters.YavSecret(
            "Secret in YAV with that can access yt. Use #key for key",
            description="Ex: sec-01djczhs4be8m2mk32w2m8mzda#nirvana-secret",
            required=True,
            # default="sec-01djczhs4be8m2mk32w2m8mzda#nirvana-secret"
        )

        with sdk2.parameters.Group("Calculation Parameters") as calc_params:
            date_range = sdk2.parameters.List(
                "Date range to calculate zC",
                description="(YYYYMMDD[HH][..YYYYMMDD[HH]] (sorted list))",
                value_type=sdk2.parameters.String,
                # default="2021071210..2021071213"
            )
            override_input_table = sdk2.parameters.String(
                "Use this table instead of regular logs",
                description="overrides date range"
            )
            product_types = sdk2.parameters.List(
                "Product Types",
                description="only {} are supported".format(", ".join(PRODUCT_TYPES)),
                required=True,
                value_type=sdk2.parameters.String,
                default=["media-creative-reach", "video-creative-reach"]
            )
            experiments = sdk2.parameters.List(
                "Experiments",
                description="format: {}".format(", ".join("{}XXXXXX".format(prefix) for prefix in sorted(EXP_PREFIXES.keys()))),
                required=True,
                value_type=sdk2.parameters.String,
                # default=["ab382699", "ab382700", "ab382701"]
            )
            etalon = sdk2.parameters.String(
                "Etalon experiment",
                description="Use same format and same experiments type as in experiments",
                # default="ab382695"
            )
            full_traffic_zc = sdk2.parameters.Bool(
                "Calculate zC for both experiments and etalon against all traffic",
                default=True
            )
            keys = sdk2.parameters.List(
                "Key fields to form group",
                required=True, default="OrderID",
                value_type=sdk2.parameters.String
            )
            max_groups = sdk2.parameters.Integer(
                "Maximum number of groups",
                required=True, default=1 << 30
            )
            conditions = sdk2.parameters.List(
                "Event conditions (grep only events that satisfy these)",
                description="Use == for true conditions and != for false conditions",
                required=False,
                value_type=sdk2.parameters.String
            )
            actions = sdk2.parameters.List(
                "Action formulas",
                required=True, default=["lclick=1.0", "complete=1.0"],
                value_type=sdk2.parameters.String,
            )
            weight_field = sdk2.parameters.String(
                "Weight field",
                description="Field in logs to use as weight for hits; if empty - all hits as weighted as 1"
            )
            calculate_cost = sdk2.parameters.Bool(
                "Calculate cost",
                description="Calculate cost and zc * cost for all experiments (etalon must be specified)",
                default=True
            )
            bid_field = sdk2.parameters.String(
                "Auction bid value field",
                description="Field in logs with final bid value",
                default="RealCost"
            )
            bill_field = sdk2.parameters.String(
                "Billed value field",
                description="Field in logs with money billed to client",
                default="EventCost"
            )

            with sdk2.parameters.RadioGroup("Commerce") as commerce:
                commerce.values[COMMERCE_ALL] = commerce.Value("All hits")
                commerce.values[COMMERCE_ONLY_COMMERCE] = commerce.Value("Only commercial hits", default=True)
                commerce.values[COMMERCE_ONLY_NOT_COMMERCE] = commerce.Value("Only non-commercial hits")

        with sdk2.parameters.Group("Calculation Parameters") as apc_check_calc_params:
            with sdk2.parameters.RadioGroup("zC calculation method") as zc_method:
                possible_methods = iter(ZC_APC_METHODS)
                method = next(possible_methods)
                zc_method.values[method] = zc_method.Value(method.upper(), default=True)

                for method in possible_methods:
                    zc_method.values[method] = zc_method.Value(method.upper())

            zc_min_conversion_rate = sdk2.parameters.Float(
                "Minimal conversion rate for the group",
                default=0.0005
            )
            zc_min_group_size = sdk2.parameters.Integer(
                "Minimal group size",
                default=1
            )
            zc_bootstrap_iterations = sdk2.parameters.Integer(
                "Bootstrap iterations",
                default=1000
            )

        with sdk2.parameters.Group("Execution Parameters") as exec_params:
            yt_pool = sdk2.parameters.String(
                "YT pool",
                required=True, default="ads-research"
            )
            yt_proxy = sdk2.parameters.String(
                "YT proxy",
                required=True, default="hahn"
            )
            yt_data_weight = sdk2.parameters.Integer(
                "YT data weight for premap stage (MB)",
                required=False, default=1 << 11
            )
            yt_working_dir = sdk2.parameters.String(
                "YT directory to work in",
                required=False,
            )
            subtasks_retries = sdk2.parameters.Integer(
                "Max subtasks retries",
                default=5
            )

        with sdk2.parameters.Output():
            zc_result_html = sdk2.parameters.String("zC (html table)")
            zc_result_wiki = sdk2.parameters.String("zC (wiki format)")

    def on_save(self):
        if not self.Parameters.override_input_table:
            if self.Parameters.date_range:
                for dt in self.Parameters.date_range:
                    if not DT_REGEXPR.match(dt):
                        raise errors.TaskFailure("Datetime range {} doesn't match pattern".format(dt))
            else:
                raise errors.TaskFailure("Either date range or input table must be specified")

        if not all(pt in PRODUCT_TYPES for pt in self.Parameters.product_types):
            raise errors.TaskFailure("Following product type are nor permitted: {}".format(
                ", ".join("\"{}\"".format(pt) for pt in self.Parameters.product_types if pt not in PRODUCT_TYPES)
            ))

        if not all(RE_EXP.match(exp) for exp in self.Parameters.experiments):
            raise errors.TaskFailure("Can't parse experiment testid")

        if self.Parameters.etalon and not RE_EXP.match(self.Parameters.etalon):
            raise errors.TaskFailure("Can't parse etalon testid")

        if not (self.Parameters.full_traffic_zc or self.Parameters.etalon):
            raise errors.TaskFailure("Can't calculate zC on experimental traffic only without etalon")

        if self.Parameters.calculate_cost and not RE_EXP.match(self.Parameters.etalon):
            raise errors.TaskFailure("Can't calculate experiments cost without etalon")

        assert self.Parameters.subtasks_retries >= 1, "Can't have non-positive number of subtasks retries"
        self.Context.tmp_yt_dir = ""

    def on_execute(self):
        with self.memoize_stage.tables_preparation():
            self._prepare_tables()

        with self.memoize_stage.spawning_subtasks(self.Parameters.subtasks_retries):
            self._spawn_children()

        with self.memoize_stage.gathering_results():
            self._gather_results()

        logging.info("Execution completed")

    def on_finish(self, prev_status, status):
        logging.info("Clearing YT working dir")
        yt_client = get_yt_client(self.Parameters)
        yt_client.remove(self.Context.tmp_yt_dir, recursive=True)

    @sdk2.header(title="zC calculations results")
    def report(self):
        if self.Parameters.zc_result_html:
            return "\n".join((
                HTML_STYLE,
                "\n".join(
                    (
                        "<h2>Calculated zC</h2>",
                        "<h2><b>WARNING! Double-check calculated cost with ABshnitsa!</b></h2>"
                        if self.Parameters.calculate_cost else "",
                        self.Parameters.zc_result_html
                    )
                )
            ))
        else:
            return "<h2>Calculations not finished yet</h2>"

    def _prepare_tables(self):
        logging.info("Starting tables preparation stage")
        binary_path = get_binary_path(self.Parameters.binary_resource, AdsReachZcCalculatorBinaryV2)
        yt_client = get_yt_client(self.Parameters)
        workdir = self.Parameters.yt_working_dir or "/".join((DEFAULT_WORKING_DIR, self.author))
        self.Context.tmp_yt_dir = make_random_yt_dir(yt_client, (workdir, ))

        if self.Context.tmp_yt_dir is None:
            raise errors.TaskFailure("Can't proceed without YT storage")

        cmd = self._get_arguments_for_premap(binary_path=binary_path, tmp_yt_dir=self.Context.tmp_yt_dir)
        env_vars = {
            "YT_PROXY": self.Parameters.yt_proxy,
            "YT_TOKEN": self.Parameters.secret_name.data()[self.Parameters.secret_name.default_key]
        }
        logging.info("Running tables preparation")
        logging.info("Command to run: {}".format(" ".join(cmd)))

        with sdk2.helpers.ProcessLog(self, logger="zc_calculator") as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr, env=env_vars)

        logging.info("Tables preparation finished, collecting common data")
        etalon_exp = None
        exp_tables = dict()

        for table in yt_client.list(self.Context.tmp_yt_dir, absolute=True):
            exp = get_yt_table_attr(table, ATTR_EXPID, yt_client)
            action = get_yt_table_attr(table, ATTR_ACTION, yt_client)

            if get_yt_table_attr(table, ATTR_IS_CONTROL, yt_client):
                if etalon_exp is None:
                    etalon_exp = exp
                elif exp != etalon_exp:
                    raise errors.TaskFailure(
                        "Found two different etalon experiments: {} and {}".format(exp, etalon_exp)
                    )

            exp_tables[(exp, action)] = table

        exps, actions = set(exp for (exp, _) in exp_tables.keys()), set(action for (_, action) in exp_tables.keys())

        if self.Parameters.etalon and etalon_exp is None:
            raise errors.TaskFailure("Failed to find any table with active etalon flag")

        self.Context.etalon_dl_subtasks = utils.encode_dict({action: 0 for action in actions})
        self.Context.apc_check_subtasks = utils.encode_dict(
            {(exp, action): 0 for (exp, action) in itertools.product(exps, actions)}
        )
        self.Context.calculate_cost_subtask = 0
        self.Context.exp_tables = utils.encode_dict(exp_tables)
        self.Context.etalon_exp = etalon_exp or -1

    def _get_arguments_for_premap(self, binary_path, tmp_yt_dir):
        cmd = [
            binary_path,
            "--exec-mode", MODE_PREMAP,
            "--tables-destination", tmp_yt_dir
        ]

        if self.Parameters.override_input_table:
            cmd.extend(("--force-input-table", self.Parameters.override_input_table))
        else:
            cmd.extend(itertools.chain(("--range", ), self.Parameters.date_range))

        cmd.append("--experiments")
        cmd.extend(self.Parameters.experiments)
        cmd.append("--product-types")
        cmd.extend(self.Parameters.product_types)
        cmd.append("--keys")
        cmd.extend(self.Parameters.keys)
        cmd.append("--action-coefficients")
        cmd.extend(self.Parameters.actions)
        cmd.extend(("--max-groups", str(self.Parameters.max_groups)))
        cmd.extend((
            "--bid-field", self.Parameters.bid_field,
            "--bill-field", self.Parameters.bill_field
        ))

        if self.Parameters.etalon:
            cmd.extend(["--etalon", self.Parameters.etalon])

        if self.Parameters.full_traffic_zc:
            cmd.append("--full-traffic-zc")

        if self.Parameters.conditions:
            cmd.append("--conditions")
            cmd.extend(self.Parameters.conditions)

        if self.Parameters.commerce == "only_commerce":
            cmd.append("--commerce")
        elif self.Parameters.commerce == "only_not_commerce":
            cmd.append("--not-commerce")

        if self.Parameters.weight_field:
            cmd.extend(("--weight-field", self.Parameters.weight_field))

        cmd.extend([
            "--yt-pool", self.Parameters.yt_pool,
            "--yt-proxy", self.Parameters.yt_proxy,
            "--log-level", "debug"
        ])

        if self.Parameters.yt_data_weight:
            cmd.extend(["--yt-data-weight", str(self.Parameters.yt_data_weight)])

        return cmd

    def _spawn_children(self):
        logging.info("Staring spawning children loop")
        logging.info("Decoding context")

        etalon_dl_subtasks = utils.decode_dict(self.Context.etalon_dl_subtasks)
        apc_check_subtasks = utils.decode_dict(self.Context.apc_check_subtasks)
        calculate_cost_subtask_id = self.Context.calculate_cost_subtask
        exp_tables = utils.decode_dict(self.Context.exp_tables)
        actions = set(action for (_, action) in exp_tables.keys())

        if self.Parameters.full_traffic_zc or self.Context.etalon_exp == -1:
            etalon_exp = CONTROL_EXP_ID
        else:
            etalon_exp = self.Context.etalon_exp

        logging.info("Checking etalon data download subtasks")
        yt_client = get_yt_client(self.Parameters)
        wait_for_dl, wait_for_apc, wait_for_cc = [], [], []

        for action in actions:
            restart = False

            if etalon_dl_subtasks[action] == 0:
                logging.info("No subtask found for action {}, spawning new one".format(action))
                restart = True
            else:
                subtask_id = etalon_dl_subtasks[action]
                subtask_status = get_task_by_id(subtask_id).status
                logging.info("Found subtask {} for action {}".format(subtask_id, action))

                if subtask_status == ctt.Status.SUCCESS:
                    logging.info("Subtask successfully finished")
                else:
                    logging.info("Subtask status is {}, restarting".format(subtask_status))
                    restart = True

            if restart:
                subtask = self._create_table_download_subtask(exp_tables[(etalon_exp, action)], action, yt_client)
                etalon_dl_subtasks[action] = subtask.id
                wait_for_dl.append(subtask)
                subtask.enqueue()

        logging.info("Checking apc_check subtasks")

        for ((exp, action), table) in exp_tables.items():
            if exp != etalon_exp:
                restart = False

                if apc_check_subtasks[(exp, action)] == 0:
                    logging.info("No apc_check subtasks found for exp {} and action {}, spawning new one".format(
                        exp, action
                    ))
                    restart = True
                else:
                    subtask_id = apc_check_subtasks[(exp, action)]
                    subtask = get_task_by_id(subtask_id)
                    logging.info("Found subtask {} for exp {} and action {}".format(subtask.id, exp, action))

                    if subtask.status == ctt.Status.SUCCESS:
                        logging.info("Subtask successfully finished")
                    elif subtask.status in ctt.Status.Group.BREAK or subtask.status in (ctt.Status.FAILURE, ctt.Status.DELETED):
                        logging.info("Subtask finished unsuccessfully, restarting")
                        restart = True
                    else:
                        if etalon_dl_subtasks[action] != subtask.Parameters.data_download_task.id:
                            logging.info("Subtask is still running with faulty data download task, restarting")
                            restart = True
                        else:
                            logging.info("Subtask is running normally")
                            wait_for_apc.append(subtask)

                if restart:
                    etalon_dl_subtask = get_task_by_id(etalon_dl_subtasks[action])
                    subtask = self._create_apc_check_subtask(table, exp, action, etalon_dl_subtask, yt_client)
                    apc_check_subtasks[(exp, action)] = subtask.id
                    wait_for_apc.append(subtask)
                    subtask.enqueue()

        if self.Parameters.calculate_cost:
            logging.info("Checking calculate_cost subtask")
            restart = False

            if calculate_cost_subtask_id == 0:
                logging.info("calculate_cost subtask not found, spawning new one")
                restart = True
            else:
                subtask = get_task_by_id(calculate_cost_subtask_id)
                logging.info("Found subtask {}".format(subtask.id))

                if subtask.status == ctt.Status.SUCCESS:
                    logging.info("Subtask successfully finished")
                elif subtask.status in ctt.Status.Group.BREAK or subtask.status in (ctt.Status.FAILURE, ctt.Status.DELETED):
                    logging.info("Subtask finished unsuccessfully, restarting")
                    restart = True

            if restart:
                subtask = self._create_calculate_cost_subtask(self.Context.tmp_yt_dir)
                self.Context.calculate_cost_subtask = subtask.id
                wait_for_cc.append(subtask)
                subtask.enqueue()

        self.Context.etalon_dl_subtasks = utils.encode_dict(etalon_dl_subtasks)
        self.Context.apc_check_subtasks = utils.encode_dict(apc_check_subtasks)

        if wait_for_dl:
            logging.info("Waiting for all etalon data download subtasks")
            raise sdk2.WaitTask(wait_for_dl, (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK), wait_all=True)
        elif wait_for_apc:
            logging.info("Waiting for all apc_check subtasks")
            raise sdk2.WaitTask(wait_for_apc, (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK), wait_all=True)
        elif wait_for_cc:
            logging.info("Waiting for calculate_cost subtask")
            raise sdk2.WaitTask(wait_for_cc, (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK), wait_all=True)
        else:
            logging.info("All subtasks successfully finished")

    def _create_table_download_subtask(self, table, action, yt_client):
        logging.info("Will download table {} as etalon data for action {}".format(table, action))
        table_sz_gb = get_yt_table_attr(table, "data_weight", yt_client) / float(1 << 30)
        task_disk_space_gb = int(math.ceil(max(table_sz_gb * 1.2, table_sz_gb + 2.0)))
        parallel_reading_threads = int(min(32, max(1, math.ceil(table_sz_gb))))
        logging.info("Table size is {} GiB, subtask disk space is {} GiB, YT reading threads {}".format(
            table_sz_gb, task_disk_space_gb, parallel_reading_threads
        ))

        logging.info("Creating table download subtask")
        subtask = AdsReachZcReadPremappedTableFromYt(
            self,
            __requirements__={"disk_space": task_disk_space_gb << 10},
            description="Etalon data download for action {}".format(action),
            yt_table=table, yt_proxy=self.Parameters.yt_proxy, secret_name=self.Parameters.secret_name,
            kill_timeout=int(math.ceil(task_disk_space_gb / 4.0 * 60 * 60)),
            parallel_reading_threads=parallel_reading_threads,
            table_size_gb=table_sz_gb
        )
        subtask.save()
        logging.info("Successfully created table download subtask for table {}".format(table))
        return subtask

    def _create_apc_check_subtask(self, table, exp, action, etalon_dl_task, yt_client):
        if not yt_client.exists(table):
            raise errors.TaskFailure("YT table {} does not exist".format(table))

        logging.info("Getting attributes from {}".format(table))
        etalon_table_size_gb = etalon_dl_task.Context.table_size_gb
        task_disk_space_gb = int(math.ceil(max(1.25 * 1.1 * etalon_table_size_gb, 1.25 * etalon_table_size_gb + 4.0)))
        ram_requirement_mb = max(32, int(math.ceil(etalon_table_size_gb * 1.25)))
        logging.info("Creating subtask, etalon data weight is {} GiB, task disk space is {} GiB".format(
            etalon_table_size_gb, task_disk_space_gb
        ))
        subtask = AdsRunApcCheckOnPreparedTable(
            self,
            __requirements__={"disk_space": task_disk_space_gb << 10, "ram": ram_requirement_mb << 10},
            description="Subtask of zC calculation task {} for exp {} and action {}".format(self.id, exp, action),
            owner=self.owner,
            binary_resource=self.Parameters.binary_resource,
            secret_name=self.Parameters.secret_name,
            premapped_table=table,
            data_download_task=etalon_dl_task,
            zc_method=self.Parameters.zc_method,
            zc_min_conversion_rate=self.Parameters.zc_min_conversion_rate,
            zc_min_group_size=self.Parameters.zc_min_group_size,
            zc_bootstrap_iterations=self.Parameters.zc_bootstrap_iterations,
            yt_proxy=self.Parameters.yt_proxy,
            kill_timeout=int(math.ceil(task_disk_space_gb / 3.0 * 60 * 60)),
            action_view=get_yt_table_attr(table, ATTR_ACTION_VIEW, yt_client)
        )
        subtask.save()
        logging.info("Successfully created apc_check subtask for table {}".format(table))
        return subtask

    def _create_calculate_cost_subtask(self, directory):
        subtask = AdsReachZcCalculateLimitedExpStats(
            self,
            __requirements__={"disk_space": 1024, "ram": 1024},
            description="Subtask of zC calculation task {}".format(self.id),
            owner=self.owner,
            binary_resource=self.Parameters.binary_resource,
            secret_name=self.Parameters.secret_name,
            yt_directory=directory,
            yt_proxy=self.Parameters.yt_proxy,
            yt_pool=self.Parameters.yt_pool,
            kill_timeout=2 * 60 * 60,
        )
        subtask.save()
        logging.info("Successfully created calculate_cost subtask")
        return subtask

    def _gather_results(self):
        logging.info("Gathering results from subtasks")
        apc_check_subtasks = utils.decode_dict(self.Context.apc_check_subtasks)
        etalon_exp = None if self.Context.etalon_exp == -1 else self.Context.etalon_exp
        control_exp = CONTROL_EXP_ID if self.Parameters.full_traffic_zc else etalon_exp
        result = dict()

        for ((exp, action), subtask_id) in apc_check_subtasks.items():
            if exp != control_exp:
                subtask = get_task_by_id(subtask_id)

                if subtask.status != ctt.Status.SUCCESS:
                    logging.info(
                        "Subtask {} for experiment {} and action {} did not finish successfully (status {})".format(
                            subtask.id, exp, action, subtask.status
                        ))
                    result[(int(exp), action)] = (None, None, subtask.Context.action_view)
                else:
                    logging.info("Subtask {} for experiment {} and action {} finished successfully".format(
                        subtask.id, exp, action
                    ))
                    result[(int(exp), action)] = (
                        subtask.Parameters.zc_mean, subtask.Parameters.zc_std, subtask.Context.action_view
                    )

        logging.info("zC results: {}".format(", ".join("{}: {}".format(k, v) for (k, v) in result.items())))
        exp_stats = None

        if self.Parameters.calculate_cost:
            if self.Context.calculate_cost_subtask:
                subtask = get_task_by_id(self.Context.calculate_cost_subtask)

                if subtask.status != ctt.Status.SUCCESS:
                    logging.info("Calculate cost subtask did not finish successfully (status {})".format(subtask.status))
                else:
                    logging.info("Calculate cost subtask finished successfully")

                    try:
                        exp_stats = json.loads(subtask.Parameters.exp_stats)
                        logging.info("Calculates experiment stats: {}".format(exp_stats))
                    except ValueError:
                        logging.error("Failed to decode subtask output: {}".format(subtask.Parameters.exp_stats))
            else:
                logging.error("Calculating cost is requested, but there is no subtask for calculation, possibly a bug :(")

        self.Parameters.zc_result_html = print_zc(
            result, exp_stats, etalon_exp if etalon_exp and self.Parameters.full_traffic_zc else None, True
        )
        self.Parameters.zc_result_wiki = print_zc(
            result, exp_stats, etalon_exp if etalon_exp and self.Parameters.full_traffic_zc else None, False
        )
        logging.info("Gathered results")
