import logging
import time

from sandbox import sdk2
from sandbox.sandboxsdk import environments as envs
from sandbox.common import errors

from sandbox.projects.ads.eshow.common.utils import get_yt_client
from sandbox.projects.ads.eshow.calculate_reach_zc.lib.yttools import read_yt_table_to_file

READ_RETRY_PAUSE = 60
READ_RETRY_MULT = 2


class AdsReachZcEtalonData(sdk2.Resource):
    """Binary to make premapped tables for zC calculation"""
    pass


class AdsReachZcReadPremappedTableFromYt(sdk2.Task):
    """Master-task to calculate reach zC"""

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1 << 10
        ram = 2 << 10
        environments = [envs.PipEnvironment("yandex-yt")]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 4 * 60 * 60

        yt_table = sdk2.parameters.String(
            "YT table to read",
            required=True
        )
        secret_name = sdk2.parameters.YavSecret(
            "Secret in YAV with access to YT. Use #key for key",
            required=True
        )

        with sdk2.parameters.Group("YT parameters"):
            yt_proxy = sdk2.parameters.String(
                "YT proxy",
                default="hahn"
            )
            parallel_reading_threads = sdk2.parameters.Integer(
                "Threads for parallel YT table reading",
                default=1
            )
            max_retries = sdk2.parameters.Integer(
                "Reading retries",
                default=3
            )

        with sdk2.parameters.Output():
            dst_resource = sdk2.parameters.Resource(
                "Resource with YT table data",
                resource_type=AdsReachZcEtalonData,
            )

    def on_save(self):
        assert self.Parameters.parallel_reading_threads >= 1, "Can't have negative amount of threads"

    def on_execute(self):
        logging.info("Starting execution")
        yt_client = get_yt_client(self.Parameters, config=self._build_yt_config())

        if not yt_client.exists(self.Parameters.yt_table):
            raise errors.TaskFailure("YT table {} does not exist".format(self.Parameters.yt_table))

        logging.info("Checked table existence, registering resource")
        self.Parameters.dst_resource = AdsReachZcEtalonData(
            self,
            "Etalon data", "etalon_data.tsv",
            ttl=1
        )
        logging.info("Starting download")
        tries, success = 0, False
        to_sleep = READ_RETRY_PAUSE

        while not success and tries < self.Parameters.max_retries:
            try:
                read_yt_table_to_file(
                    self.Parameters.yt_table,
                    str(sdk2.ResourceData(self.Parameters.dst_resource).path),
                    yt_client,
                    header=True, append=False, unordered=True
                )
            except Exception as exc:
                logging.error("Failed to read table, got exception: {}".format(exc))
                tries += 1

                if tries < self.Parameters.max_retries:
                    logging.info("Sleeping for {} seconds before next retry".format(to_sleep))
                    time.sleep(to_sleep)
                    to_sleep *= READ_RETRY_MULT
                else:
                    logging.error("Failed to read table after {} retries".format(tries))
                    raise errors.TaskFailure("Can't read table")
            else:
                success = True

        sdk2.ResourceData(self.Parameters.dst_resource).ready()
        logging.info("Execution completed")

    def _build_yt_config(self):
        yt_conf = {"read_retries": {"enable": False}}

        if self.Parameters.parallel_reading_threads > 1:
            yt_conf["read_parallel"] = {
                "enable": True,
                "max_thread_count": self.Parameters.parallel_reading_threads
            }

        return yt_conf
