import logging
import random
import string
import time

from sandbox import sdk2
from sandbox.common import errors


def get_binary_path(resource, resource_type):
    if not resource:
        logging.info("No specific binary given - will find latest released\n")
        latest_released = sdk2.Resource.find(
            type=resource_type,
            attrs={"released": "stable"}
        ).first()

        if latest_released:
            logging.info("Using resource {}\n".format(latest_released.id))
            return str(sdk2.ResourceData(latest_released).path)
        else:
            raise errors.TaskFailure("Failed to find at least one released binary")
    else:
        return str(sdk2.ResourceData(resource).path)


def get_task_by_id(task_id):
    return sdk2.Task.find(id=task_id).order(-sdk2.Task.id).first()


def get_yt_client(params, config=None):
    from yt.wrapper import YtClient

    return YtClient(
        proxy=params.yt_proxy,
        token=params.secret_name.data()[params.secret_name.default_key],
        config=config
    )


def get_yt_table_attr(table, attr, yt_client):
    from yt.wrapper import ypath_join

    return yt_client.get(ypath_join(table, "@{}".format(attr)))


def make_html_table(header, columns, rows):
    res = ["<div class=\"table\">"]

    if header:
        res.append("<h2>{}</h2>".format(header))

    res.extend(["<table>", "<thead>", "<tr>"])

    for column in columns:
        res.append("<th>{}</th>".format(column))

    res.extend(["</tr>", "</thead>", "<tbody>"])

    for row in rows:
        res.append("<tr>")

        for column in columns:
            res.append("<td>{}</td>".format(row.get(column, "unknown")))

        res.append("</tr>")

    res.extend(["</tbody>", "</table>", "</div>"])
    return "\n".join(res)


def make_random_string(ln=16, randomize_with_time=True):
    if randomize_with_time:
        random.seed(hash(time.time()))

    return "".join(random.choice(string.ascii_letters) for _ in range(ln))


def make_random_yt_dir(yt_client, path, ln=32, retries=10):
    from yt.wrapper import ypath_join

    logging.info("Creating temp YT directory")
    path_chunks = []

    if isinstance(path, str):
        path_chunks.append(path)
    else:
        path_chunks.extend(path)

    flag = True
    tries = 0
    joined_path = ypath_join(*(path_chunks + [make_random_string(ln)]))

    while tries < retries and flag:
        if yt_client.exists(joined_path):
            logging.error("Node {} already exists, will try once more".format(joined_path))
            tries += 1
            joined_path = ypath_join(*(path_chunks + [make_random_string(ln)]))
        else:
            flag = False
            yt_client.create("map_node", joined_path, recursive=True)

    if tries < retries:
        logging.info("Created directory {}".format(joined_path))
        return joined_path
    else:
        logging.error("Failed to create temp directory after {} tries".format(retries))
        return None
