import os
import six

import sandbox.projects.common.constants as consts

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.yav import Secret
from sandbox.common.types.client import Tag
from sandbox.projects.common.build.YaMake2 import YaMake2
from sandbox.common.utils import chain
from sandbox.common.types.resource import State
from sandbox.common.types.task import Status, ReleaseStatus
from sandbox.projects.common.vcs import arc

from sandbox.projects.ads.infra.ads_infra_release_builder.resources import LogosBinary, LogosMonoNileUdf, AdsLogosDocs, LogosGraphDoc  # noqa
from sandbox.projects.ads.infra.ads_infra_release_builder.lib import parse_svn_revision, parse_project

TOOL_ENV_VAR = "AQINFRA_TOOL"
GRAPH_DOC_TOOL = "GRAPH_DOC"
DISABLE_COLOR = "DISABLE_COLOR"

ROBOT_LOGOS_PREPROD_SECRET_UUID = "sec-01egg35h0qjbh0jc03ey7e6tpf"
ROBOT_LOGOS_PREPROD_SECRET_KEY = "nirvana-secret"
ROBOT_LOGOS_PROD_SECRET_UUID = "sec-01daxhrvak3kbcd6fxhydp3esp"
ROBOT_LOGOS_PROD_SECRET_KEY = "secret"


class AdsLogosBuilder(sdk2.Task):
    """ Logos Builder """

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 4096
        client_tags = Tag.GENERIC & ~Tag.INTEL_E5645  # exclude hosts without AVX support

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 3600  # 10 hours
        description = "Logos Graph Manager"
        mono_bin = sdk2.parameters.String("Mono bin", description="Arcadia path to logos mono binary to build", default="logos/projects/ads/graph/bin/logos_tool", required=True)
        mono_nile_udf = sdk2.parameters.String("Mono nile udf", description="Arcadia path to logos mono nile udf to build", required=False)
        old_release_params_switch = sdk2.parameters.Bool("Switch to old format args", description="Exmp: Use update_graph if True else update-graph", default=False)
        logos_secret = sdk2.parameters.YavSecret(
            "Logos secret",
            description="Logos secret https://oauth.yandex-team.ru/authorize?response_type=token&client_id=dbb4bcf6e5b44640b16b4ec626d7952d Use #key for key."
        )
        update_after_switch = sdk2.parameters.Bool("Update new graph w.r.t. release metainfo", default=True)
        dry_run = sdk2.parameters.Bool("Dry run without call execution", default=False)
        build_binary = sdk2.parameters.Bool("Build binary", default=True)
        build_graph_doc = sdk2.parameters.Bool("Build graph doc", description="Choose to build LOGOS_GRAPH_DOC as output resource", default=True)
        with build_binary.value[True]:
            project_name = sdk2.parameters.String("ProjectName", description="Postfix of created branch", default="ads")
            test = sdk2.parameters.Bool("Run tests", description="Run tests while building mono binary", default=True)
            use_arc_fuse = sdk2.parameters.Bool("Use Arc VCS", description="Choose to use arc fuse api", default=True)
            with use_arc_fuse.value[False]:  # building svn
                revision = sdk2.parameters.String("Revision", description="Arcadia trunk revision")
            with use_arc_fuse.value[True]:  # building arc
                with sdk2.parameters.RadioGroup("Branch or revision", per_line=2, description="Use branch or revision") as i_have:
                    i_have.values["branch"] = i_have.Value(value="Branch")
                    i_have.values["revision"] = i_have.Value(value="Revision", default=True)
                with i_have.value["branch"]:
                    branch = sdk2.parameters.String(
                        "Arcadia branch / commit hash / revision",
                        description="Examples: r7312366, users/dim-gonch/logos, trunk, fd6aa52239c, releases/logos/release-2020-06-16",
                        required=True
                    )
                with i_have.value["revision"]:
                    create_branch = sdk2.parameters.Bool(
                        "Create logos release branch before running YA_MAKE",
                        description="Creates release branch based on revison number before running YA MAKE. releases/logos/release-r<REVISION>",
                        default=False
                    )
                    arc_revision = sdk2.parameters.String("Revision", description="Arcadia trunk revision (get from arc info)", required=True)
        with build_binary.value[False]:
            logos_binary_resource = sdk2.parameters.Resource(
                "Logos binary",
                description="Resource of type LOGOS_BINARY, built by YA MAKE task",
                required=True
            )
            udf_resource_id = sdk2.parameters.Resource(
                "Logos binary of mono nile udf",
                description="Resource of type LOGOS_MONO_NILE_UDF, built by YA MAKE task"
            )

        with sdk2.parameters.Group("Deploy to Reactor") as deploy_block:
            with sdk2.parameters.RadioGroup("Deploy to", per_line=3, description="Choose environment system to deploy") as deploy_to:
                deploy_to.values["prod"] = deploy_to.Value(value="Prod")
                deploy_to.values["preprod"] = deploy_to.Value(value="Preprod")
                deploy_to.values["no_deploy"] = deploy_to.Value(value="No deploy", default=True)
            with deploy_to.value["prod"], deploy_to.value["preprod"]:
                release_id = sdk2.parameters.String("Release id", description="Reactor graph release-id name", do_not_copy=True)
            switch_release = sdk2.parameters.Bool("Switch release", default=False, do_not_copy=True)

    # compatable param, delete in future
    def _get_arg_param_old(self, param):
        if not self.Parameters.old_release_params_switch:
            return param

        if param.startswith("--"):
            return "--" + param[2:].replace('-', '_')

        return param.replace('-', '_')

    def _run_build_task(self, checkout_arcadia_from_url, target, result_resource):
        target_dir = os.path.dirname(target)
        custom_fields = dict(
            test=self.Parameters.test,
            result_single_file=True,
            use_aapi_fuse=self.Parameters.use_arc_fuse,
            aapi_fallback=True,
            use_arc_instead_of_aapi=False,
            checkout_arcadia_from_url=checkout_arcadia_from_url,
            targets=";".join([target_dir]),
            arts=target,
            arts_source="",
            result_rt=str(result_resource),
            result_rd="{} for {}\n{}".format(result_resource.__name__, self.id, self.Parameters.description),
            build_system=consts.SEMI_DISTBUILD_BUILD_SYSTEM,
            build_type="release",
            clear_build=False,
            check_return_code=True,
            strip_binaries=True,
            ya_yt_store=True,
            sandbox_tags="GENERIC & ~INTEL_E5645",
            result_attrs={"project": parse_project(target)}
        )
        task = self.server.task({
            "type": str(YaMake2),
            "description": "Build {} for {}\n{}".format(result_resource.__name__, self.id, self.Parameters.description),
            "owner": self.owner,
            "notifications": self.server.task[self.id].read()["notifications"],
            "priority": {
                "class": self.Parameters.priority.cls,
                "subclass": self.Parameters.priority.scls,
            },
            "children": True,
            "custom_fields": [{"name": n, "value": v} for n, v in six.iteritems(custom_fields)]
        })
        task_id = task["id"]
        self.server.batch.tasks.start.update([task_id])
        return task_id

    def _build_graph_doc(self, project, arcadia_revision, binary_path):
        svn_revision = parse_svn_revision(arcadia_revision)
        release_name = "preprod" if arcadia_revision.startswith("releases/logos/r") else "trunk"
        released = ReleaseStatus.PRESTABLE if release_name == "preprod" else ReleaseStatus.TESTING

        logos_json_resource = LogosGraphDoc(
            self, self.Parameters.description, "latest.json",
            arcadia_revision=svn_revision,
            release_name=release_name,
            project=project,
            released=released
        )
        logos_json_resource_data = sdk2.ResourceData(logos_json_resource)
        env = os.environ.copy()
        env[TOOL_ENV_VAR] = GRAPH_DOC_TOOL
        env[DISABLE_COLOR] = "1"
        with sdk2.helpers.ProcessLog(self, logger="logos_binary_build_doc") as pl:
            sp.check_call([
                unicode(binary_path),
                "--{}".format(release_name),
                "--path",
                unicode(logos_json_resource_data.path)
            ], stdout=pl.stdout, stderr=sp.STDOUT, env=env)
        logos_json_resource_data.ready()
        self.Context.logos_graph_doc_id = logos_json_resource.id

    def call_with_dry_run(self, args, stdout, stderr, env):
        if not self.Parameters.dry_run:
            sp.check_call(args, stdout=stdout, stderr=stderr, env=env)
        else:
            sp.check_call(
                ["echo", "Making a call with args: {}\n".format(" ".join([str(arg) for arg in args]))],
                stdout=stdout, stderr=stderr, env=env
            )

    def on_execute(self):

        if self.Parameters.logos_secret:
            logos_token = self.Parameters.logos_secret.data()[self.Parameters.logos_secret.default_key]
        elif self.Parameters.deploy_to == "prod":
            logos_token = Secret(ROBOT_LOGOS_PROD_SECRET_UUID).data()[ROBOT_LOGOS_PROD_SECRET_KEY]
        elif self.Parameters.deploy_to == "preprod":
            logos_token = Secret(ROBOT_LOGOS_PREPROD_SECRET_UUID).data()[ROBOT_LOGOS_PREPROD_SECRET_KEY]
        elif self.Parameters.dry_run:
            logos_token = "FAKE_DRY_RUN_LOGOS_TOKEN"
        elif self.Parameters.create_branch:
            raise RuntimeError("No logos token provided. Use prod, preprod or specify logos_secret manualy")

        logos_binary_resource = None
        if not self.Parameters.build_binary:
            logos_binary_resource = self.Parameters.logos_binary_resource
            if self.Parameters.udf_resource_id:
                logos_binary_resource.nile_udf_resource_id = self.Parameters.udf_resource_id
        elif self.Context.build_task_id:
            built_resource_id = LogosBinary.find(
                task_id=self.Context.build_task_id,
                state=State.READY,
            ).first().id
            logos_binary_resource = sdk2.Resource[built_resource_id]
            if self.Context.build_udf_task_id:
                built_udf_resource_id = LogosMonoNileUdf.find(
                    task_id=self.Context.build_udf_task_id,
                    state=State.READY,
                ).first().id
                logos_binary_resource.nile_udf_resource_id = built_udf_resource_id

        if logos_binary_resource is None:
            if not self.Parameters.use_arc_fuse:
                checkout_arcadia_from_url = "arcadia:/arc/trunk/arcadia@" + (self.Parameters.revision or "")
            else:
                if self.Parameters.i_have == "revision":
                    svn_rev = "r{}".format(self.Parameters.arc_revision)
                    if self.Parameters.create_branch:
                        branch = "releases/logos/{}_{}".format(self.Parameters.project_name, svn_rev)
                        arc_cli = arc.Arc(arc_oauth_token=logos_token)
                        with arc_cli.mount_path(None, svn_rev, fetch_all=False) as arcadia_src_dir:
                            try:
                                arc_cli.branch(arcadia_src_dir, branch=branch, remote_branch=branch)
                                raise RuntimeError("Branch {} already exists. Use 'branch' option to specify.".format(branch))
                            except arc.ArcCommandFailed:
                                pass
                            arc_cli.checkout(arcadia_src_dir, branch=branch, create_branch=True)
                            arc_cli.push(arcadia_src_dir, upstream=branch)
                        checkout_arcadia_from_url = "arcadia-arc:/#" + branch
                    else:
                        branch = svn_rev
                elif self.Parameters.i_have == "branch":
                    branch = self.Parameters.branch
                checkout_arcadia_from_url = "arcadia-arc:/#" + branch

            self.Context.build_task_id = self._run_build_task(checkout_arcadia_from_url, self.Parameters.mono_bin, LogosBinary)
            build_tasks = [self.Context.build_task_id]
            if self.Parameters.mono_nile_udf:
                self.Context.build_udf_task_id = self._run_build_task(checkout_arcadia_from_url, self.Parameters.mono_nile_udf, LogosMonoNileUdf)
                build_tasks.append(self.Context.build_udf_task_id)
            raise sdk2.WaitTask(
                build_tasks, set(chain(Status.Group.FINISH, Status.Group.BREAK)),
                wait_all=True
            )
        else:
            self.Context.logos_binary_resource_id = logos_binary_resource.id
            built_resource_path = sdk2.ResourceData(logos_binary_resource).path
            deploy_to = self.Parameters.deploy_to

            if self.Parameters.build_graph_doc:
                project = parse_project(unicode(self.Parameters.mono_bin))
                self._build_graph_doc(
                    project,
                    logos_binary_resource.arcadia_revision,
                    built_resource_path
                )
            try:
                svn_revision = parse_svn_revision(logos_binary_resource.arcadia_revision)
            except RuntimeError:
                if self.Parameters.release_id:
                    svn_revision = None
                else:
                    raise
            if deploy_to in ("prod", "preprod"):
                command_env = os.environ.copy()
                command_env["LOGOS_TOKEN"] = logos_token

                additional_args = []

                release_id = self.Parameters.release_id if self.Parameters.release_id else "release_" + svn_revision

                description = "Task: https://sandbox.yandex-team.ru/task/{id}\nRevision: {rev}".format(
                    id=self.id, rev=(svn_revision if svn_revision else release_id)
                )
                if self.Parameters.description:
                    description = description + "\n" + self.Parameters.description
                additional_args += ["--description", description]
                if deploy_to == "prod" or (deploy_to == "preprod" and self.Parameters.switch_release):
                    additional_args += ["--release-candidate"]
                if deploy_to == "preprod":
                    additional_args += ["--builtin-subgraph"]
                with sdk2.helpers.ProcessLog(self, logger="logos_binary_update_graph") as pl:
                    self.call_with_dry_run(
                        [
                            unicode(built_resource_path),
                            "--prod" if deploy_to == "prod" else "--preprod",
                            self._get_arg_param_old("update-graph"),
                            self._get_arg_param_old("--sandbox-resource"),
                            str(logos_binary_resource.id),
                            self._get_arg_param_old("--release-id"),
                            release_id,
                            "--meta-info",
                            "release_meta_info.json"
                        ] + additional_args,
                        stdout=pl.stdout,
                        stderr=sp.STDOUT,
                        env=command_env
                    )
                self.set_info("Created graph with release id: {}".format(release_id))
                self.Context.logos_release_id = release_id
            if deploy_to == "prod" and self.Parameters.switch_release:
                with sdk2.helpers.ProcessLog(self, logger="logos_binary_switch_release") as pl:
                    self.call_with_dry_run(
                        [
                            unicode(built_resource_path),
                            "--prod",
                            "switch-release"
                        ],
                        stdout=pl.stdout,
                        stderr=sp.STDOUT,
                        env=command_env
                    )
                if self.Parameters.update_after_switch:
                    with sdk2.helpers.ProcessLog(self, logger="logos_update_after_switch") as pl:
                        self.call_with_dry_run(
                            [
                                unicode(built_resource_path),
                                "--prod",
                                "bootstrap-release",
                                "--meta-info",
                                "release_meta_info.json",
                                self._get_arg_param_old("--release-id"),
                                release_id,
                            ],
                            stdout=pl.stdout,
                            stderr=sp.STDOUT,
                            env=command_env
                        )
                if not self.Parameters.dry_run:
                    self.server.release(
                        task_id=logos_binary_resource.task_id,
                        type=ReleaseStatus.STABLE,
                        subject=self.Parameters.description
                    )
            elif deploy_to == "preprod":
                if self.Parameters.switch_release:
                    with sdk2.helpers.ProcessLog(self, logger="logos_binary_switch_release") as pl:
                        self.call_with_dry_run(
                            [
                                unicode(built_resource_path), "--preprod", "switch-release"
                            ],
                            stdout=pl.stdout,
                            stderr=sp.STDOUT,
                            env=command_env
                        )
                if self.Parameters.update_after_switch:
                    with sdk2.helpers.ProcessLog(self, logger="logos_update_after_switch") as pl:
                        self.call_with_dry_run(
                            [
                                unicode(built_resource_path),
                                "--preprod",
                                "bootstrap-release",
                                "--meta-info",
                                "release_meta_info.json",
                                self._get_arg_param_old("--release-id"),
                                release_id,
                            ],
                            stdout=pl.stdout,
                            stderr=sp.STDOUT,
                            env=command_env
                        )
                if not self.Parameters.dry_run:
                    self.server.release(
                        task_id=logos_binary_resource.task_id,
                        type=ReleaseStatus.TESTING,
                        subject=self.Parameters.description
                    )
