import datetime as dt
import logging
import six

import typing as tp

from sandbox import sdk2
from sandbox.common.types.task import Status, ReleaseStatus
from sandbox.common.types.client import Tag
from sandbox.common.utils import chain
from sandbox.projects.common.arcadia.sdk import mount_arc_path, mounted_path_svnversion
from sandbox.projects.ads.infra.logos_release_watchman.lib import AdsLogosReleaseWatchman, AdsLogosReleaseSchedule
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.yav import Secret


ROBOT_LOGOS_PREPROD_SECRET_UUID = "sec-01egg35h0qjbh0jc03ey7e6tpf"
ROBOT_LOGOS_PREPROD_SECRET_KEY = "nirvana-secret"


def make_run_tasks_cli(
    run_tasks_args,  # type: tp.Optional[str]
    run_date,  # type: dt.datetime
    logos_release_id,  # type: str
    schedule_file,  # type: str
):
    day_before_last_date = run_date - dt.timedelta(days=1)
    if run_tasks_args is None:
        run_tasks_args = "--builtin-subgraph" \
                         " --expand-to-external" \
                         " --last-date {:%Y-%m-%d}T00:00:00" \
                         " --prod-inputs AdsJoinedEFHColumns={:%Y-%m-%d}T23:00:00 AdsFlatVisitV2LogStream5Min={:%Y-%m-%d}T23:55:00"

        run_tasks_args = run_tasks_args.format(run_date, day_before_last_date, day_before_last_date)
    return [
        "--preprod", "run-tasks",
        "--release-id", logos_release_id,
        "--schedule-file", schedule_file
    ] + run_tasks_args.split()


class AdsLogosPreprodRunner(sdk2.Task):
    class Requirements(sdk2.Requirements):
        client_tags = Tag.GENERIC & ~Tag.INTEL_E5645  # exclude hosts without AVX support

    class Parameters(sdk2.Task.Parameters):
        description = "Task which creates release branch, starts and waits for preprod. Then builds doc and creates draft of prod builder."
        run_tasks_args = sdk2.parameters.String(
            "Arguments for run_tasks command",
            description="./logos_tool --preprod run-tasks --release-id <GENERATED_BY_TASK> $run_tasks_args",
        )
        whitelist = sdk2.parameters.String(
            "Whitelist in json format",
            description="See logos/projects/ads/graph/preprod_whitelist.json for example"
        )
        max_failed_instances = sdk2.parameters.Integer("Max failed instances", default=None)
        max_runtime = sdk2.parameters.Integer("Max release tuntime in seconds", default=None)
        poll_release_timeout = sdk2.parameters.Integer("Interval between polls in seconds", default=None)
        revision = sdk2.parameters.String("SVN revision for running preprod", default="")

    def _get_from_task_context(
        self,
        task_id,
        var
    ):
        ctx = self.server.task[task_id].context.read()
        return ctx[var]

    def _create_child_task(
        self,
        task_type,
        description,
        kill_timeout,
        custom_fields
    ):
        task = self.server.task({
            "type": task_type,
            "description": description,
            "owner": self.owner,
            "notifications": self.server.task[self.id].read()["notifications"],
            "kill_timeout": kill_timeout,
            "priority": {
                "class": self.Parameters.priority.cls,
                "subclass": self.Parameters.priority.scls,
            },
            "children": True,
            "custom_fields": [{"name": n, "value": v} for n, v in six.iteritems(custom_fields)]
        })
        task_id = task["id"]
        return task_id

    def _ensure_task_success(
        self,
        task_id
    ):
        task = self.server.task[task_id].read()
        if task["status"] != str(Status.SUCCESS):
            raise RuntimeError("Child task {}: {} was not finished succesfully, can't proceed".format(
                task["type"], task["id"]
            ))

    def _upload_schedule(self, schedule_file):
        resource = AdsLogosReleaseSchedule(
            self,
            "Preprod schedule for release {}".format(self.Context.logos_release_id),
            schedule_file
        )
        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()
        return resource.id

    def _run_logos_builder_preprod(self, secret):
        if not self.Context.preprod_logos_builder_id:
            revision = self.Parameters.revision
            if not revision:
                token = secret.data()[secret.default_key]
                with mount_arc_path("arcadia-arc:/#trunk", use_arc_instead_of_aapi=True, arc_oauth_token=token) as arcadia:
                    repo_info = mounted_path_svnversion(arcadia, arc_vcs=True)
                    logging.info("Using trunk, info {}".format(repo_info))
                    revision = repo_info["revision"]
            custom_fields = {
                "build_binary": True,
                "build_graph_doc": False,
                "use_arc_fuse": True,
                "test": True,
                "i_have": "revision",
                "create_branch": True,
                "deploy_to": "preprod",
                "arc_revision": revision,
            }
            task_id = self._create_child_task(
                task_type="ADS_LOGOS_BUILDER",
                description="Build preprod for revision {}".format(revision),
                kill_timeout=60 * 60 * 3,
                custom_fields=custom_fields
            )
            self.Context.preprod_logos_builder_id = task_id
            self.server.batch.tasks.start.update([task_id])
            raise sdk2.WaitTask(
                [task_id], set(chain(Status.Group.FINISH, Status.Group.BREAK)),
                wait_all=True
            )

    def _run_tasks(
        self,
        secret,
        schedule_file,
        logos_binary_resource_id,
        logos_release_id
    ):
        logos_binary_resource = sdk2.Resource[logos_binary_resource_id]
        logos_binary_resource_path = str(sdk2.ResourceData(logos_binary_resource).path)

        run_date = dt.datetime.now() - dt.timedelta(days=1)
        run_tasks_args = self.Parameters.run_tasks_args
        if not run_tasks_args:
            run_tasks_args=None
        command_line = [logos_binary_resource_path] + make_run_tasks_cli(
            run_tasks_args,
            run_date=run_date,
            logos_release_id=logos_release_id,
            schedule_file=schedule_file
        )

        logos_binary_env = {
            "LOGOS_TOKEN": secret.data()[secret.default_key]
        }
        with sdk2.helpers.ProcessLog(self, logger="logos_binary") as pl:
            sp.check_call(command_line, stdout=pl.stdout, stderr=sp.STDOUT, env=logos_binary_env)

    def _run_preprod_and_wait(self, secret, logos_binary_resource_id, logos_release_id):
        if not self.Context.release_watchman_id:
            schedule_file = "preprod-schedule"
            self._run_tasks(secret, schedule_file, logos_binary_resource_id, logos_release_id)
            schedule_resource_id = self._upload_schedule(schedule_file)
            custom_fields = {
                "release_env":          "preprod",
                "mail_to":              self.author,
                "schedule_file":        schedule_resource_id,
                "logos_binary":         logos_binary_resource_id,
                "logos_secret":         str(secret),
            }
            for param in (
                "max_failed_instances",
                "max_runtime",
                "poll_release_timeout",
                "whitelist"
            ):
                param_value = getattr(self.Parameters, param, None)
                if param_value is not None:
                    custom_fields[param] = param_value

            kill_timeout = 60 * 60 * 24
            if self.Parameters.max_runtime is not None:
                kill_timeout = max(kill_timeout, self.Parameters.max_runtime)

            task_id = self._create_child_task(
                task_type=AdsLogosReleaseWatchman.name,
                description="Wait preprod for release {}".format(logos_release_id),
                kill_timeout=kill_timeout,
                custom_fields=custom_fields
            )
            self.Context.release_watchman_id = task_id
            self.server.batch.tasks.start.update([task_id])
            raise sdk2.WaitTask(
                [task_id], set(chain(Status.Group.FINISH, Status.Group.BREAK)),
                wait_all=True
            )

    def _run_logos_builder_graph_doc(self, logos_binary_resource_id, logos_release_id):
        if not self.Context.prod_graph_doc_builder_id:
            custom_fields = {
                "build_binary": False,
                "build_graph_doc": True,
                "logos_binary_resource": logos_binary_resource_id,
                "deploy_to": "no_deploy",
                "release_id": logos_release_id,
            }
            task_id = self._create_child_task(
                task_type="ADS_LOGOS_BUILDER",
                description="Build graph doc for release {}".format(logos_release_id),
                kill_timeout=60 * 60 * 10,
                custom_fields=custom_fields
            )
            self.Context.prod_graph_doc_builder_id = task_id
            self.server.batch.tasks.start.update([task_id])
            raise sdk2.WaitTask(
                [task_id], set(chain(Status.Group.FINISH, Status.Group.BREAK)),
                wait_all=True
            )

    def _draft_logos_builder_prod(self, logos_binary_resource_id, logos_release_id, logos_graph_doc_id):
        custom_fields = {
            "build_binary": False,
            "build_graph_doc": False,
            "logos_binary_resource": logos_binary_resource_id,
            "deploy_to": "prod",
            "release_id": logos_release_id,
            "switch_release": True,
            "logos_graph_doc_id": logos_graph_doc_id,
        }
        self._create_child_task(
            task_type="ADS_LOGOS_BUILDER",
            description="Upload prod for release {}".format(logos_release_id),
            kill_timeout=60 * 60 * 3,
            custom_fields=custom_fields
        )

    def on_execute(self):
        if self.Parameters.run_tasks_args:
            if "release-id" in self.Parameters.run_tasks_args:
                raise RuntimeError("Please, don'y use release-id in run_tasks_args. It is automatically generated.")
        secret = Secret(ROBOT_LOGOS_PREPROD_SECRET_UUID, default_key=ROBOT_LOGOS_PREPROD_SECRET_KEY)
        with self.memoize_stage.build_preprod:
            self._run_logos_builder_preprod(secret)
        self._ensure_task_success(self.Context.preprod_logos_builder_id)
        logos_binary_resource_id = self._get_from_task_context(self.Context.preprod_logos_builder_id, "logos_binary_resource_id")
        logos_release_id = self._get_from_task_context(self.Context.preprod_logos_builder_id, "logos_release_id")

        with self.memoize_stage.run_preprod_and_wait:
            self._run_preprod_and_wait(secret, logos_binary_resource_id, logos_release_id)
        self._ensure_task_success(self.Context.release_watchman_id)
        logos_report_id = self._get_from_task_context(self.Context.release_watchman_id, "logos_report_id")

        with self.memoize_stage.attach_report:
            with open(str(sdk2.ResourceData(sdk2.Resource[logos_report_id]).path)) as report:
                self.Context.report = report.read().strip()
            self.Context.save()

        with self.memoize_stage.mark_logos_binary_prestable:
            logos_binary_resource = sdk2.Resource[logos_binary_resource_id]
            self.server.release(
                task_id=logos_binary_resource.task_id,
                type=ReleaseStatus.PRESTABLE,
                subject="Logos binary release candidate"
            )

        with self.memoize_stage.build_doc:
            self._run_logos_builder_graph_doc(logos_binary_resource_id, logos_release_id)
        self._ensure_task_success(self.Context.prod_graph_doc_builder_id)
        logos_graph_doc_id = self._get_from_task_context(self.Context.prod_graph_doc_builder_id, "logos_graph_doc_id")

        with self.memoize_stage.build_prod:
            self._draft_logos_builder_prod(logos_binary_resource_id, logos_release_id, logos_graph_doc_id)

    @sdk2.footer(title="Preprod report.html")
    def report(self):
        return self.Context.report or "Report will appear after ADS_LOGOS_RELEASE_WATCHMAN success"
