import os
import json
from logging import info

from sandbox import sdk2
from sandbox.projects.common import task_env


class AdsProductionModels(sdk2.Resource):
    """ Plaintext file resource """
    pass


class FactorsDashboard(sdk2.Task):

    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Settings') as settings_block:
            dashboards_binary = sdk2.parameters.Resource('Dashboards binary', required=True)
            conf = sdk2.parameters.JSON("Dashboards configuration", required=True)
        with sdk2.parameters.Group('Mysql') as mysql_block:
            mysql_host = sdk2.parameters.String("Host", required=True, default="bsbackup02i.yandex.ru")
            mysql_port = sdk2.parameters.Integer("Port", required=True, default=3306)
            mysql_user = sdk2.parameters.String("User", required=True, default="sandbox")
            mysql_db = sdk2.parameters.String("Db", required=True, default="yabsdb")
            mysql_vault_owner = sdk2.parameters.String("Vault password owner", required=True, default="ML-ENGINE")
            mysql_vault_name = sdk2.parameters.String(
                "Vault password name",
                required=True,
                default="bsbackup_mysql_password",
            )
        with sdk2.parameters.Group('Grafana') as grafana_block:
            grafana_host = sdk2.parameters.String("Host", required=True, default="https://grafana.yandex-team.ru")
            grafana_vault_owner = sdk2.parameters.String("Vault token owner", required=True, default="BSMR")
            grafana_vault_name = sdk2.parameters.String("Vault token name", required=True, default="grafana_token")

    def get_env_with_vault_data(self):
        mysql_password = sdk2.Vault.data(self.Parameters.mysql_vault_owner, self.Parameters.mysql_vault_name)
        grafana_token = sdk2.Vault.data(self.Parameters.grafana_vault_owner, self.Parameters.grafana_vault_name)

        env = os.environ.copy()
        env['GRAFANA_TOKEN'] = grafana_token
        env['MYSQL_PASSWORD'] = mysql_password

        return env

    def get_binary_path(self):
        return str(sdk2.ResourceData(self.Parameters.dashboards_binary).path)

    def on_execute(self):

        conf_file_name = 'dashboards.conf'
        with open(conf_file_name, 'w') as f:
            f.write(json.dumps(self.Parameters.conf))

        production_models_file_name = "ads_production_models.json"

        cmd = [
            self.get_binary_path(),
            "--conf-file", conf_file_name,
            "--grafana-host", self.Parameters.grafana_host,
            "--mysql-host", self.Parameters.mysql_host,
            "--mysql-port", str(self.Parameters.mysql_port),
            "--mysql-user", self.Parameters.mysql_user,
            "--mysql-db", self.Parameters.mysql_db,
            "--production-models-file", production_models_file_name
        ]

        info('Running command {}'.format(cmd))

        with sdk2.helpers.ProcessLog(self, logger='compute_metrics') as pl:
            stdout = sdk2.helpers.subprocess.Popen(
                cmd,
                env=self.get_env_with_vault_data(),
                stdout=sdk2.helpers.subprocess.PIPE,
                stderr=pl.stderr
            ).communicate()[0]

        result_info = []
        for line in sorted(stdout.strip().split("\n")):
            name, url = line.split(" : ")
            result_info.append("<a href='{}'>Dashboard {}</a>".format(url, name))

        resource = AdsProductionModels(self, 'Current ads production models', production_models_file_name)
        sdk2.ResourceData(resource).ready()

        self.set_info("\n".join(result_info), do_escape=False)
