# -*- coding: utf-8 -*-

import time
from datetime import timedelta, datetime
from logging import error, info
import simplejson

import sandbox.sdk2
from sandbox.projects.common.yabs.graphite import Graphite, five_min_metric, YABS_SERVERS


def get_latest_resource(*args, **kwargs):
    resource = sandbox.sdk2.Resource['LM_DUMPS_LIST'].find(state='READY', *args, **kwargs).order(-sandbox.sdk2.Resource.id).first()
    if resource:
        return resource
    error("Can't find latest resource: %s", kwargs)


def parse_last_log_date(date_str):
    # Ask @fram why this is a bad idea
    if len(date_str) == len("201806220800"):
        return datetime.strptime(date_str, '%Y%m%d%H%M')
    if "-" in date_str:
        return datetime.strptime(date_str, '%Y-%m-%d')
    else:
        if len(date_str) == len('2018060709'):
            return datetime.strptime(date_str, '%Y%m%d%H')
    return datetime.strptime(date_str, '%Y%m%d')


def make_dates(file_name):
    js = simplejson.load(open(file_name, 'r'))
    model_to_date = {}
    for model in js:
        model_to_date[model['LinearModelID']] = parse_last_log_date(model['LastLogDate'])
    return model_to_date


class ModelDateMonitor2(sandbox.sdk2.Task):
    class Requirements(sandbox.sdk2.Requirements):
        cores = 1
        ram = 1024
        disk_space = 5 * 1024

        class Caches(sandbox.sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sandbox.sdk2.Parameters):
        graphite_hosts = sandbox.sdk2.parameters.String("Comma separated list of all bsgraphite hosts",
                                                        default=",".join(YABS_SERVERS))
        metric_prefix = sandbox.sdk2.parameters.String("metric prefix for graphite", required=True)

    def get_meta_resource(self):
        return get_latest_resource(
            attrs={"location": "meta"}
        )

    def get_stat_resource(self):
        return get_latest_resource(
            attrs={"location": "stat", "shard": 1}
        )

    def on_execute(self):
        graphite_hosts = self.Parameters.graphite_hosts.split(",")
        graphite_obj = Graphite(graphite_hosts, timedelta(seconds=5).total_seconds())

        meta_resource = self.get_meta_resource()
        stat_resource = self.get_stat_resource()

        stat_filename = str(sandbox.sdk2.ResourceData(stat_resource).path)
        meta_filename = str(sandbox.sdk2.ResourceData(meta_resource).path)

        model_to_date = make_dates(meta_filename)
        model_to_date.update(make_dates(stat_filename))

        for model, last_log_date in model_to_date.iteritems():
            delta = datetime.now() - last_log_date
            info("Processing model {model}".format(model=model))
            info("Now time is {dt}".format(dt=datetime.now()))
            info("DateStr from dump {last_log_date}".format(last_log_date=last_log_date))
            info("Delta from dump {delta}".format(delta=delta))
            metric_prefix = "{}.linear_model_{}".format(self.Parameters.metric_prefix, model)
            metric_prefix = five_min_metric(metric_prefix)
            info(metric_prefix)
            value = delta.total_seconds()
            graphite_obj.send([(metric_prefix, value, time.time())])

        metric_prefix = "{}.lm_dumps_list".format(self.Parameters.metric_prefix)
        metric_prefix = five_min_metric(metric_prefix)
        info("DateStr from LM_DUMPS_LIST {meta_resource_created}".format(meta_resource_created=meta_resource.created))
        info(metric_prefix)
        delta = datetime.now() - meta_resource.created.replace(tzinfo=None)
        delta = delta - timedelta(hours=3)  # This is UTC shift, do not know now to do better way
        info("Delta from meta resource {delta}".format(delta=delta))
        value = delta.total_seconds()
        graphite_obj.send([(metric_prefix, value, time.time())])
