from datetime import datetime, timedelta
import hashlib
import logging
import os
import requests
import subprocess

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types.task import ReleaseStatus

from sandbox.projects.ads.common import AmMrUtils
from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.yabs.qa.utils.yt_utils import set_yt_node_ttl


class AdsQuotaWatcher(BaseBinTask):
    """calculate quota usage diff"""
    class Parameters(BaseBinTask.Parameters):
        resource_attrs = BaseBinTask.Parameters.resource_attrs(default={"task_type": "ADS_QUOTA_WATCHER"})
        release_version = BaseBinTask.Parameters.release_version(default=ReleaseStatus.STABLE)
        output_prefix = sdk2.parameters.String(
            "path to temporary tables",
            required=True,
            default="//home/ads/stat/diff_quota"
        )
        quota_stat_path = sdk2.parameters.String(
            "path to quota stats",
            required=True,
            default="//home/ads/stat/quota"
        )
        yt_cluster = sdk2.parameters.String(
            'YT cluster',
            default='hahn',
        )
        secret_name = sdk2.parameters.YavSecret(
            'yav secret id, stores data from https://oauth.yandex-team.ru/client/15dc39db788b44ae82c2418be8b3bf0b',
            default='sec-01fdeqs70j5m05r8604bf5adzg'
        )
        analyze_config = sdk2.parameters.JSON(
            'list of checks: tuples (<timedelta(days=x)>,dict with restrictions',
            default='[[1, {"disk": 50000000000000}], [7, {"disk": 150000000000000}]]'
        )
        chats_to_notify = sdk2.parameters.List(
            'List of chats in JNS to notify',
        )

    def get_out_directory(self, yt_client):
        out_directory = '{}/{}'.format(self.Parameters.output_prefix, self.id)
        yt_client.remove(out_directory, recursive=True, force=True)
        yt_client.create('map_node', out_directory, recursive=True)
        return out_directory

    def notify(self, session, message):
        def format_message(message):
            data = {
                "project": "Ads",
                "params": {
                    "message": {
                        "string_value": message,
                    }
                },
                "template": "default",
                "request_id": hashlib.md5(message.encode()).hexdigest(),
                "recipient": {
                    "telegram": {
                        "chat_name": self.Parameters.chats_to_notify
                    }
                },
                "abc_service": "ads"
            }
            return data
        response = session.post('https://jns.yandex-team.ru/api/messages/send', json=format_message(message))
        data = {
            "code": response.status_code,
            "json": response.json()
        }
        logging.info("JNS response [{code}]: {json}".format(**data))

    @staticmethod
    def get_current_table(quota_stat_path):
        new_date = datetime.now().strftime("%Y%m%d")
        new_table = '{}/{}'.format(quota_stat_path, new_date)
        return new_table

    def iter_one_interval(self, yt_client, quota_stat_path, session, time_delta, thresholds):
        from ads.daily_duty.libs.py_quota_pinger import build_stat, analyze_stat
        new_date = datetime.now().strftime("%Y%m%d")
        new_table = self.get_current_table(quota_stat_path)
        old_date = (datetime.now()-timedelta(days=time_delta)).strftime("%Y%m%d")
        old_table = '{}/{}'.format(quota_stat_path, old_date)
        out_directory = self.get_out_directory(yt_client)
        out_table = '{}/{}'.format(out_directory, "diff_{}_{}".format(old_date, new_date))
        comparator_resource = AmMrUtils.find(
            attrs={"released": "stable"}
        ).first()
        if not comparator_resource:
            raise errors.TaskFailure("Could not find binary")
        data = sdk2.ResourceData(comparator_resource)

        has_been_calced = yt_client.exists(out_table)
        logging.info("checking previous table {out}: {flag}".format(out=out_table, flag=has_been_calced))
        if has_been_calced:
            return []

        path = data.path
        cmd = [
            str(path),
            'diff',
            '-s', 'hahn',
            '--old', old_table,
            '--new', new_table,
            '--dst', out_table,
            '--sort-by', 'path',
            '--sort-by', 'owner',
            '--sort-by', 'account',
            '--diff-mode', 'full']
        logging.info(cmd)
        with sdk2.helpers.ProcessLog(self, logger='upload') as pl:
            env = os.environ.copy()
            env["YT_TEMP_DIR"] = "//home/bs/tmp"
            co = subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout, env=env)
        logging.info("am_mr_utils exit code: {co}".format(co=co))

        set_yt_node_ttl(out_table, 86400, yt_client)

        stat = build_stat(yt_client, out_table)
        return analyze_stat(session, stat, time_delta, thresholds)

    def on_execute(self):
        import yt.wrapper as yt
        from ads.daily_duty.libs.py_quota_pinger import (
            message_for_top_users,
            AccountInfo, QuotaStat, StaffResolver,
        )
        logging.info(self.Parameters.secret_name.data().keys())

        token = self.Parameters.secret_name.data()['oauth_token']
        os.environ['YT_TOKEN'] = token
        yt_client = yt.YtClient(token=token, proxy=self.Parameters.yt_cluster)
        quota_stat_path = self.Parameters.quota_stat_path
        session = requests.Session()
        session.headers['Authorization'] = "OAuth {}".format(token)

        staff_resolver = StaffResolver(session)
        messages = set()
        quota_stat = QuotaStat.from_yt_lazy(yt_client, self.get_current_table(quota_stat_path))
        for account in ["bs", "ads"]:
            account_info = AccountInfo.from_yt(yt_client, account)
            logging.info("Current disk quota usage for {acc}: {usage}".format(
                acc=account, usage=account_info.disk_usage_ratio))
            logging.info("Current chunk quota usage for {acc}: {usage}".format(
                acc=account, usage=account_info.chunk_usage_ratio))
            if account_info.disk_usage_ratio > 0.97:
                top_users_messages = message_for_top_users(
                    resource="disk",
                    staff_resolver=staff_resolver,
                    top_users=quota_stat.get_top_for_acc(account, "disk", size=10),
                    quota_name=account,
                    usage=account_info.disk_usage_ratio,
                )
                messages.update(top_users_messages)

                for (time_delta, thresholds) in self.Parameters.analyze_config:
                    res = self.iter_one_interval(yt_client, quota_stat_path, session, time_delta, thresholds)
                    logging.info(res)
                    messages.update(res)
            elif account_info.chunk_usage_ratio > 0.95:
                top_users_messages = message_for_top_users(
                    resource="chunks",
                    staff_resolver=staff_resolver,
                    top_users=quota_stat.get_top_for_acc(account, "chunks", size=10),
                    quota_name=account,
                    usage=account_info.chunk_usage_ratio,
                )
                messages.update(top_users_messages)
        logging.info("Analysis is done, about to send {num} messages".format(num=len(messages)))
        for msg in messages:
            logging.info(msg)
            self.notify(session, msg)

        return
