# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
import sandbox.common.types.resource as ctr

from sandbox.projects.ads.common import YangConversionBannersBin

from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.adv_machine.common import process_wrapper
from sandbox.projects.geobase.Geodata5BinStable import resource as gbr


logger = logging.getLogger(__name__)


class SendGroupedFactorLogToYang(sdk2.Task):

    """Send GroupedFactorLogToYang"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 24 * 60 * 60
        with sdk2.parameters.Group('Assessment type parameters') as assessment_type_params_block:
            banner_conversion = sdk2.parameters.Bool('Banner Conversion', default=True)
            synonym = sdk2.parameters.Bool('Synonymous', default=False)

        with sdk2.parameters.Group('Grep parameters') as grep_params_block:
            start_date = sdk2.parameters.String("Start date (FORMAT: YYYYMMDD)", required=True)
            num_of_days = sdk2.parameters.Integer("Number of days", required=True, default=1)
            send_priority = sdk2.parameters.Integer("SendPriority from 1 to 5", required=True, default=1)
            num_of_banners_per_bin = sdk2.parameters.Integer("Number of banners per bin", required=True, default=13)
            max_queries_per_banner = sdk2.parameters.Integer("Max queries per banner", required=True, default=5)
            feature_set_id = sdk2.parameters.Integer('Feature set id', required=True)
            formula_id = sdk2.parameters.Integer('Formula id', required=False, default=0)
            bins_count = sdk2.parameters.Integer('Bins count', required=True, default=20)
            banners_table = sdk2.parameters.String("Banners table", required=True, default='//home/advquality/adv_machine/robot/latest/STREAMS.MERGED_FINAL.4mr_beta')

        with sdk2.parameters.Group('Recipients parameters') as recipient_block:
            emails = sdk2.parameters.String('A comma separated list of emails', default='rudolf@yandex-team.ru')
            tickets = sdk2.parameters.String('A comma separated list of startrek tickets')
            solomon_project = sdk2.parameters.String('Solomon project, leave this field empty, if you don\'t want to send data to Solomon')
            solomon_cluster = sdk2.parameters.String('Solomon cluster, leave this field empty, if you don\'t want to send data to Solomon')
            solomon_service = sdk2.parameters.String('Solomon service, leave this field empty, if you don\'t want to send data to Solomon')

        with sdk2.parameters.Group('YT parameters') as yt_block:
            yt_proxy = sdk2.parameters.String('YT proxy', required=True, default='hahn')
            yt_token_vault = sdk2.parameters.String('YT_TOKEN vault name', required=True, default='adv_machine_yt_token')
            yt_pool = sdk2.parameters.String('YT_POOL', required=True, default='adv-machine-research')
            yang_table_path = sdk2.parameters.String('Yang processed dump table', required=True, default='//home/advquality/ang/ang_processed_dump')
            dst_path = sdk2.parameters.String('Dst path (don\'t change if you\'re not sure!)', required=True, default='//home/advquality/data_for_yang/')

        with sdk2.parameters.Group('Yang parameters') as yang_params_block:
            dst_yang_prefix = sdk2.parameters.String('Dst yang grefix (don\'t change if you\'re not sure!)', required=True, default='//home/advquality/conversion_banners/ang_request/conversion/')
            dst_yang_synonym_prefix = sdk2.parameters.String('Dst yang synonym grefix (don\'t change if you\'re not sure!)',
                                                             required=True, default='//home/advquality/conversion_banners/ang_request/synonyms/')
            num_of_lines_in_part = sdk2.parameters.Integer("Number of lines in one part (yang)", required=True, default=10000)
            error_rate = sdk2.parameters.Float("Permissible error rate", required=True, default=0.1)

        with sdk2.parameters.Group('Tags') as task_tags:
            additional_tags = sdk2.parameters.String('Additional tags, delimited by comma')

        with sdk2.parameters.Group('Resources') as resources_block:
            yang_conversion_banners_res_id = sdk2.parameters.LastReleasedResource(
                'yang_conversion_banners binary resource',
                resource_type=YangConversionBannersBin,
                state=(ctr.State.READY, ),
                required=True,
            )
            geodata = sdk2.parameters.LastReleasedResource(
                'geodata5.bin',
                resource_type=gbr.GEODATA5BIN_STABLE,
                state=(ctr.State.READY,),
                required=True,
            )

    def on_execute(self):
        bin_res = sdk2.ResourceData(self.Parameters.yang_conversion_banners_res_id)
        geodata_res = sdk2.ResourceData(self.Parameters.geodata)

        cmd = [
            str(bin_res.path), 'prepare-grouped-fl-and-send-to-yang',
            '--noBannerConversion' if not self.Parameters.banner_conversion else '',
            '--withSynonym' if self.Parameters.synonym else '',
            '-o', self.Parameters.dst_path,
            '--banners-info', self.Parameters.banners_table,
            '-G', str(geodata_res.path),
            '-D', self.Parameters.start_date,
            '--days', str(self.Parameters.num_of_days),
            '--bins-count', str(self.Parameters.bins_count),
            '--priority', str(self.Parameters.send_priority),
            '--fl-banners-per-bin', str(self.Parameters.num_of_banners_per_bin),
            '--fl-queries-per-banner', str(self.Parameters.max_queries_per_banner),
            '--fs', str(self.Parameters.feature_set_id),
            '--formula-id', str(self.Parameters.formula_id),
            '--emails', self.Parameters.emails,
            '--tickets', self.Parameters.tickets,
            '--solomon-project', self.Parameters.solomon_project,
            '--solomon-service', self.Parameters.solomon_service,
            '--solomon-cluster', self.Parameters.solomon_cluster,
            '-s', self.Parameters.yt_proxy,
            '--outYangPrefix', self.Parameters.dst_yang_prefix,
            '--outYangSynonymPrefix', self.Parameters.dst_yang_synonym_prefix,
            '--linesInPart', str(self.Parameters.num_of_lines_in_part),
            '--errorRate', str(self.Parameters.error_rate),
            '-v'
        ]

        tags = set(['grouped-factor-log'])
        tags |= set([x for x in self.Parameters.additional_tags.split(',') if x])
        cmd.extend(['--tags', ','.join(list(sorted(tags)))])

        env = {'MR_RUNTIME': 'YT'}
        if self.Parameters.yt_token_vault:
            env['YT_TOKEN'] = sdk2.Vault.data(self.Parameters.yt_token_vault)
        if self.Parameters.yt_pool:
            env['YT_POOL'] = self.Parameters.yt_pool

        with process_wrapper(self, logger='yang_conversion_banners') as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr, env=env)
