# -*- coding: utf-8 -*-
import datetime
import logging

from sandbox import common
from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.ads.send_yang_conversion_banners import SendYangConversionBanners


logger = logging.getLogger(__name__)


def generate_params(sim_distance):
    res = {}
    sd_to_type = {'25': 'search',
                  '100025': 'rm',
                  '700025': 'dsa',
                  '914': 'search-offline',
                  '100314': 'rm-offline',
                  '700330': 'dsa-offline',
                  '48': 'synonym'}
    if sim_distance in set(('25', '100025', '700025', '914', '100314', '700330')):
        res = {'agr_list': 'Source,OrderID',
               'num_of_rows_per_group_per_day': 150,
               'sim_distance': '^.*$',
               'lua_formula_fields': {
                   'Source': 'SimDistance == "{}" and "ADVMACHINE" or "Others"'.format(sim_distance)
               },
               'shows_ratio': 0,
               'banner_conversion': True,
               'synonym': False}
    elif sim_distance == '48':
        res = {'agr_list': 'OrderID',
               'num_of_rows_per_group_per_day': 300,
               'sim_distance': '^48$',
               'lua_formula_fields': None,
               'shows_ratio': 1,
               'banner_conversion': False,
               'synonym': True}
    else:
        raise Exception('Bad SimDistance')
    res['ban_request_type'] = sd_to_type[sim_distance]
    return res


class SendOrderIdToYang(sdk2.Task):

    """Задача для отправки кликов по OrderID на качество асессорам"""

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Parameters') as param_block:
            with sdk2.parameters.RadioGroup("SimDistance") as sim_distance:
                sim_distance.values['25'] = sim_distance.Value(value='broadmatch (SD=25)')
                sim_distance.values['100025'] = sim_distance.Value(value='relevance match (SD=100025)')
                sim_distance.values['700025'] = sim_distance.Value(value='dsa (SD=700025)')
                sim_distance.values['914'] = sim_distance.Value(value='broadmatch offline (SD=914)')
                sim_distance.values['100314'] = sim_distance.Value(value='relevance match offline (SD=100314)')
                sim_distance.values['700330'] = sim_distance.Value(value='dsa offline (SD=700330)')
                sim_distance.values['48'] = sim_distance.Value(value='synonym (SD=48)')
            order_ids = sdk2.parameters.List('List of OrderIDs (номер БК)', required=True)
            emails = sdk2.parameters.List('List of emails', required=True)
            start_date = sdk2.parameters.String("Start date (FORMAT: YYYYMMDD)", required=True,
                                                default=datetime.datetime.strftime(datetime.datetime.now() - datetime.timedelta(days=30), "%Y%m%d"))
            num_of_days = sdk2.parameters.Integer("Number of days", required=True, default=30)
            tickets = sdk2.parameters.List('List of startrek tickets (e.g. ADVMACHINE-1047)', required=True)

        with sdk2.parameters.Group('YT parameters') as yt:
            yt_proxy = sdk2.parameters.String('YT proxy', required=True, default='hahn')
            yt_token_vault = sdk2.parameters.String('YT_TOKEN vault name', required=True, default='adv_machine_yang_yt_token')
            yt_pool = sdk2.parameters.String('YT_POOL', required=True, default='advquality-yang')

    def on_execute(self):
        subtasks_list = list(self.find().limit(0))
        if subtasks_list:
            failed_subtasks = filter(lambda t: t.status not in ctt.Status.Group.SUCCEED, subtasks_list)
            if len(failed_subtasks):
                raise common.errors.TaskFailure('Some of subtasks have failed!')
            return
        order_ids_string = "|".join(map(lambda x: "({})".format(x), self.Parameters.order_ids))
        subtask = self._create_send_yang_conversion_banners_task(order_ids_string)
        subtask.enqueue()
        raise sdk2.WaitTask([subtask], ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=True)

    def _create_send_yang_conversion_banners_task(self, order_ids_string):
        custom_params = generate_params(self.Parameters.sim_distance)
        return SendYangConversionBanners(
            self,
            owner=self.Parameters.owner,
            priority=ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.NORMAL),
            description=self.Parameters.description,
            banner_conversion=custom_params['banner_conversion'],
            traffic_nativity=False,
            synonym=custom_params['synonym'],
            start_date=self.Parameters.start_date,
            num_of_days=self.Parameters.num_of_days,
            agr_list=custom_params['agr_list'],
            num_of_rows_per_group_per_day=custom_params['num_of_rows_per_group_per_day'],
            type_id='^1$',
            sim_distance=custom_params['sim_distance'],
            order_id='^{}$'.format(order_ids_string),
            clicks_ratio=1.0,
            shows_ratio=0.0,
            emails=','.join(self.Parameters.emails),
            tickets=','.join(['ADVMACHINE-1047'] + self.Parameters.tickets),
            save_to_yt=True,
            ban_request=True,
            ban_request_type=custom_params['ban_request_type'],
            lua_formula_fields=custom_params['lua_formula_fields'],
            yt_proxy=self.Parameters.yt_proxy,
            yt_token_vault=self.Parameters.yt_token_vault,
            yt_pool=self.Parameters.yt_pool,
            task_tags=['order_complain']
        )  # Rewrite to **custom_params after python 3.5
