# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
import sandbox.common.types.resource as ctr

from sandbox.projects.ads.common import YangConversionBannersBin

from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.adv_machine.common import process_wrapper
from sandbox.projects.geobase.Geodata5BinStable import resource as gbr


logger = logging.getLogger(__name__)


class SendShowsWithFeaturesToYang(sdk2.Task):

    """Send shows with features """

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 24 * 60 * 60
        with sdk2.parameters.Group('Assessment type parameters') as assessment_type_params_block:
            banner_conversion = sdk2.parameters.Bool('Banner Conversion', default=True)
            synonym = sdk2.parameters.Bool('Synonymous', default=False)

        with sdk2.parameters.Group('Grep parameters') as grep_params_block:
            sim_distance = sdk2.parameters.String('SimDistance regex', required=True, default='^.*$')
            order_id = sdk2.parameters.String('OrderID regex', required=True, default='^.*$')
            num_of_rows_per_group_per_day = sdk2.parameters.Integer("Number of rows per group", required=True, default=750)
            shows_ratio = sdk2.parameters.Float("Shows ratio", required=True, default=1.0)
            start_date = sdk2.parameters.String("Start date (FORMAT: YYYYMMDD)", required=True)
            num_of_days = sdk2.parameters.Integer("Number of days", required=True, default=1)
            send_priority = sdk2.parameters.Integer("SendPriority from 1 to 5", required=True, default=1)
            feature_set_id = sdk2.parameters.Integer('Feature set id', required=True)
            page_id = sdk2.parameters.String('PageID regex', required=True, default='^.*$')
            query_classifiers = sdk2.parameters.String('Query classifiers regex', required=True, default='^.*$')
            join_offer_info = sdk2.parameters.Bool("Join offer info", default=False)

        with sdk2.parameters.Group('Recipients parameters') as recipient_block:
            emails = sdk2.parameters.String('A comma separated list of emails', default='muradmazitov@yandex-team.ru')
            tickets = sdk2.parameters.String('A comma separated list of startrek tickets')
            solomon_project = sdk2.parameters.String('Solomon project, leave this field empty, if you don\'t want to send data to Solomon')
            solomon_cluster = sdk2.parameters.String('Solomon cluster, leave this field empty, if you don\'t want to send data to Solomon')
            solomon_service = sdk2.parameters.String('Solomon service, leave this field empty, if you don\'t want to send data to Solomon')

        with sdk2.parameters.Group('YT parameters') as yt_block:
            yt_proxy = sdk2.parameters.String('YT proxy', required=True, default='hahn')
            yt_token_vault = sdk2.parameters.String('YT_TOKEN vault name', required=True, default='adv_machine_yt_token')
            yt_pool = sdk2.parameters.String('YT_POOL', required=True, default='adv-machine-research')
            dst_path = sdk2.parameters.String('Dst path (don\'t change if you\'re not sure!)', required=True, default='//home/advquality/data_for_yang/')

        with sdk2.parameters.Group('Yang parameters') as yang_params_block:
            dst_yang_prefix = sdk2.parameters.String('Dst yang grefix (don\'t change if you\'re not sure!)', required=True, default='//home/advquality/conversion_banners/ang_request/conversion/')
            dst_yang_synonym_prefix = sdk2.parameters.String('Dst yang synonym grefix (don\'t change if you\'re not sure!)',
                                                             required=True, default='//home/advquality/conversion_banners/ang_request/synonyms/')
            num_of_lines_in_part = sdk2.parameters.Integer("Number of lines in one part (yang)", required=True, default=10000)
            num_of_rows_in_group = sdk2.parameters.Integer("Number of rows in one group", required=True, default=1)
            error_rate = sdk2.parameters.Float("Permissible error rate", required=True, default=0.1)

        with sdk2.parameters.Group('Tags') as task_tags:
            additional_tags = sdk2.parameters.String('Additional tags, delimited by comma')

        with sdk2.parameters.Group('Resources') as resources_block:
            yang_conversion_banners_res_id = sdk2.parameters.LastReleasedResource(
                'yang_conversion_banners binary resource',
                resource_type=YangConversionBannersBin,
                state=(ctr.State.READY, ),
                required=True,
            )
            geodata = sdk2.parameters.LastReleasedResource(
                'geodata5.bin',
                resource_type=gbr.GEODATA5BIN_STABLE,
                state=(ctr.State.READY,),
                required=True,
            )

    def on_execute(self):
        bin_res = sdk2.ResourceData(self.Parameters.yang_conversion_banners_res_id)
        geodata_res = sdk2.ResourceData(self.Parameters.geodata)

        cmd = [
            str(bin_res.path), 'send-shows-with-features',
            '--noBannerConversion' if not self.Parameters.banner_conversion else '',
            '--withSynonym' if self.Parameters.synonym else '',
            '--dstPath', self.Parameters.dst_path,
            '-G', str(geodata_res.path),
            '-D', self.Parameters.start_date,
            '--days', str(self.Parameters.num_of_days),
            '-S', self.Parameters.sim_distance,
            '-O', self.Parameters.order_id,
            '--fs', str(self.Parameters.feature_set_id),
            '--rows', str(self.Parameters.num_of_rows_per_group_per_day),
            '--sr', str(self.Parameters.shows_ratio),
            '--agrList', 'SimDistance',
            '--priority', str(self.Parameters.send_priority),
            '--emails', self.Parameters.emails,
            '--tickets', self.Parameters.tickets,
            '--solomon-project', self.Parameters.solomon_project,
            '--solomon-service', self.Parameters.solomon_service,
            '--solomon-cluster', self.Parameters.solomon_cluster,
            '-s', self.Parameters.yt_proxy,
            '--outYangPrefix', self.Parameters.dst_yang_prefix,
            '--outYangSynonymPrefix', self.Parameters.dst_yang_synonym_prefix,
            '--linesInPart', str(self.Parameters.num_of_lines_in_part),
            '--grouped', str(self.Parameters.num_of_rows_in_group),
            '--errorRate', str(self.Parameters.error_rate),
            '-P', self.Parameters.page_id,
            '--classifiers', self.Parameters.query_classifiers,
            '--join-offer-info' if self.Parameters.join_offer_info else '',
            '-v'
        ]

        tags = set(['factor-log', 'shows'])
        tags |= set([x for x in self.Parameters.additional_tags.split(',') if x])
        cmd.extend(['--tags', ','.join(list(sorted(tags)))])

        env = {'MR_RUNTIME': 'YT'}
        if self.Parameters.yt_token_vault:
            env['YT_TOKEN'] = sdk2.Vault.data(self.Parameters.yt_token_vault)
        if self.Parameters.yt_pool:
            env['YT_POOL'] = self.Parameters.yt_pool

        with process_wrapper(self, logger='yang_conversion_banners') as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr, env=env)
