# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
import sandbox.common.types.resource as ctr

from sandbox.projects.ads.common import YangConversionBannersBin

from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.adv_machine.common import process_wrapper
from sandbox.projects.geobase.Geodata5BinStable import resource as gbr


logger = logging.getLogger(__name__)


class SendYangConversionBanners(sdk2.Task):

    """Send YangConversionBanners"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 24 * 60 * 60
        with sdk2.parameters.Group('Assessment type parameters') as assessment_type_params_block:
            banner_conversion = sdk2.parameters.Bool('Banner Conversion', default=True)
            synonym = sdk2.parameters.Bool('Synonymous', default=False)
        with sdk2.parameters.Group('Grep parameters') as grep_params_block:
            start_date = sdk2.parameters.String("Start date (FORMAT: YYYYMMDD)", required=True)
            num_of_days = sdk2.parameters.Integer("Number of days", required=True, default=4)
            send_priority = sdk2.parameters.Integer("SendPriority from 1 to 5", required=True, default=1)
            agr_list = sdk2.parameters.String(
                'A comma separated list of aggregation group (may include: Date, ContextType, TypeID, SimDistance, ExperimentBits, OrderID, QueryCategory)',
                required=True,
                default='Date,SimDistance'
            )
            num_of_rows_per_group_per_day = sdk2.parameters.Integer("Number of rows per group", required=True, default=750)
            join_offer_info = sdk2.parameters.Bool("Join offer info", default=False)
            context_type = sdk2.parameters.String('ContextType regex', required=True, default='^.*$')
            type_id = sdk2.parameters.String('TypeId regex', required=True, default='^1$')
            sim_distance = sdk2.parameters.String('SimDistance regex', required=True, default='^.*$')
            experiment_bits = sdk2.parameters.String('ExperimentBits regex', required=True, default='^.*$')
            experiment_ids = sdk2.parameters.String('ExperimentIds regex', required=True, default='^.*$')
            test_ids = sdk2.parameters.List("ActiveTestIds", required=False)
            order_id = sdk2.parameters.String('OrderID regex', required=True, default='^.*$')
            page_id = sdk2.parameters.String('PageID regex', required=True, default='^.*$')
            query_classifiers = sdk2.parameters.String('Query classifiers regex', required=True, default='^.*$')
            clicks_ratio = sdk2.parameters.Float("Clicks ratio", required=True, default=1.0)
            shows_ratio = sdk2.parameters.Float("Shows ratio", required=True, default=0.0)
            force_tag = sdk2.parameters.Bool('Force tag', required=False)
            lua_formula_fields = sdk2.parameters.Dict("Lua formula fields", required=False)
            sort_by_aggr_group = sdk2.parameters.Bool('Sort by aggregation group', required=False, default=True)

        with sdk2.parameters.Group('Recipients parameters') as recipient_block:
            emails = sdk2.parameters.String('A comma separated list of emails', default='rudolf@yandex-team.ru')
            tickets = sdk2.parameters.String('A comma separated list of startrek tickets')
            solomon_project = sdk2.parameters.String('Solomon project, leave this field empty, if you don\'t want to send data to Solomon')
            solomon_cluster = sdk2.parameters.String('Solomon cluster, leave this field empty, if you don\'t want to send data to Solomon')
            solomon_service = sdk2.parameters.String('Solomon service, leave this field empty, if you don\'t want to send data to Solomon')

        with sdk2.parameters.Group('YT parameters') as yt_block:
            yt_proxy = sdk2.parameters.String('YT proxy', required=True, default='hahn')
            yt_token_vault = sdk2.parameters.String('YT_TOKEN vault name', required=True, default='adv_machine_yt_token')
            yt_pool = sdk2.parameters.String('YT_POOL', required=True, default='adv-machine-research')
            log_prefix = sdk2.parameters.String('Log Prefix', required=True, default='//home/bs/logs')
            log_type = sdk2.parameters.String('Log Type', required=True, default='JoinedEFHFactors')
            dst_path = sdk2.parameters.String('Dst path (don\'t change if you\'re not sure!)', required=True, default='//home/advquality/data_for_yang/')

        with sdk2.parameters.Group('Yang parameters') as yang_params_block:
            honeypots_type = sdk2.parameters.String('Type of honeypots (Example: cehac,fashion,all)', required=True, default='')
            dst_yang_prefix = sdk2.parameters.String('Dst yang grefix (don\'t change if you\'re not sure!)', required=True, default='//home/advquality/conversion_banners/ang_request/conversion/')
            dst_yang_synonym_prefix = sdk2.parameters.String('Dst yang synonym grefix (don\'t change if you\'re not sure!)',
                                                             required=True, default='//home/advquality/conversion_banners/ang_request/synonyms/')
            num_of_lines_in_part = sdk2.parameters.Integer("Number of lines in one part (yang)", required=True, default=10000)
            num_of_rows_in_group = sdk2.parameters.Integer("Number of rows in one group", required=True, default=1)
            error_rate = sdk2.parameters.Float("Permissible error rate", required=True, default=0.1)

        with sdk2.parameters.Group('Postprocessing Params') as postprocessing_params_block:
            save_to_yt = sdk2.parameters.Bool('save results to yt table', default=False)
            ban_request = sdk2.parameters.Bool('ban OrderID if quality is bad', default=False)
            ban_request_type = sdk2.parameters.String('ban type (rm, dsa, search, synonym)', default="")

        with sdk2.parameters.Group('Tags') as task_tags:
            with sdk2.parameters.CheckGroup('Main tags') as main_tags:
                main_tags.choices = ((elem, elem) for elem in ['experiment', 'regular'])
            additional_tags = sdk2.parameters.String('Additional tags, delimited by comma')

        with sdk2.parameters.Group('Resources') as resources_block:
            yang_conversion_banners_res_id = sdk2.parameters.LastReleasedResource(
                'yang_conversion_banners binary resource',
                resource_type=YangConversionBannersBin,
                state=(ctr.State.READY, ),
                required=True,
            )
            geodata = sdk2.parameters.LastReleasedResource(
                'geodata5.bin',
                resource_type=gbr.GEODATA5BIN_STABLE,
                state=(ctr.State.READY,),
                required=True,
            )

    def on_execute(self):
        bin_res = sdk2.ResourceData(self.Parameters.yang_conversion_banners_res_id)
        geodata_res = sdk2.ResourceData(self.Parameters.geodata)

        lua_formula_args = []
        for lua_formula_name, lua_formula_value in self.Parameters.lua_formula_fields.iteritems():
            lua_formula_args.extend(['--lua-formula-name', lua_formula_name, '--lua-formula-value', lua_formula_value])

        cmd = [
            str(bin_res.path), 'prepare-and-send-to-yang',
            '-D', self.Parameters.start_date,
            '--days', str(self.Parameters.num_of_days),
            '--priority', str(self.Parameters.send_priority),
            '--agrList', self.Parameters.agr_list,
            '--rows', str(self.Parameters.num_of_rows_per_group_per_day),
            '-C', self.Parameters.context_type,
            '-T', self.Parameters.type_id,
            '-S', self.Parameters.sim_distance,
            '-E', self.Parameters.experiment_bits,
            '-X', self.Parameters.experiment_ids,
            '-O', self.Parameters.order_id,
            '-P', self.Parameters.page_id,
            '--classifiers', self.Parameters.query_classifiers,
            '--cr', str(self.Parameters.clicks_ratio),
            '--sr', str(self.Parameters.shows_ratio),
            '--emails', self.Parameters.emails,
            '--tickets', self.Parameters.tickets,
            '--solomon-project', self.Parameters.solomon_project,
            '--solomon-service', self.Parameters.solomon_service,
            '--solomon-cluster', self.Parameters.solomon_cluster,
            '-s', self.Parameters.yt_proxy,
            '-p', self.Parameters.log_prefix,
            '-t', self.Parameters.log_type,
            '--dstPath', self.Parameters.dst_path,
            '--outYangPrefix', self.Parameters.dst_yang_prefix,
            '--honeypots-type', self.Parameters.honeypots_type,
            '--outYangSynonymPrefix', self.Parameters.dst_yang_synonym_prefix,
            '--linesInPart', str(self.Parameters.num_of_lines_in_part),
            '--grouped', str(self.Parameters.num_of_rows_in_group),
            '--errorRate', str(self.Parameters.error_rate),
            '-G', str(geodata_res.path),
            '--noBannerConversion' if not self.Parameters.banner_conversion else '',
            '--withSynonym' if self.Parameters.synonym else '',
            '--saveToYt' if self.Parameters.save_to_yt else '',
            '--banRequest' if self.Parameters.ban_request else '',
            '--banRequestType', self.Parameters.ban_request_type,
            '--sortByAggrGroup' if self.Parameters.sort_by_aggr_group else '',
            '--join-offer-info' if self.Parameters.join_offer_info else ''
            '-v'
        ]
        cmd.extend(lua_formula_args)  # TODO: rewrite to *lua_formula_args after Python 3.5 support

        test_ids_args = []
        for test_id in self.Parameters.test_ids:
            test_ids_args.extend(['--test-ids', str(test_id)])

        cmd.extend(test_ids_args)

        tags = set(['yabs-log'])
        tags |= set(self.Parameters.main_tags)
        tags |= set([x for x in self.Parameters.additional_tags.split(',') if x])
        cmd.extend(['--tags', ','.join(list(sorted(tags)))])

        env = {'MR_RUNTIME': 'YT'}
        if self.Parameters.yt_token_vault:
            env['YT_TOKEN'] = sdk2.Vault.data(self.Parameters.yt_token_vault)
        if self.Parameters.yt_pool:
            env['YT_POOL'] = self.Parameters.yt_pool

        with process_wrapper(self, logger='yang_conversion_banners') as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr, env=env)
