import logging
import os
import tarfile
import shutil

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

logger = logging.getLogger(__name__)

TMP_DIR_NAME = "./tmp_dir"


def get_resource_path(resource):
    return str(sdk2.ResourceData(resource).path)


def get_last_resource_by_params(params):
    return sdk2.Resource.find(**params).order(-sdk2.Resource.id).first()


def get_last_resource(params):
    return get_resource_path(get_last_resource_by_params(params))


def convert_resource_to_sthash_dump(resource, dump_dst, compression_type, model_format):
    tsar_dump_path = get_resource_path(resource)

    to_sthash_converter_params = {
        "type": "CONVERT_TO_STHASH_BINARY",
        "state": "READY"
    }

    if tarfile.is_tarfile(tsar_dump_path):
        logging.info('Unpacking tar file')
        tar = tarfile.open(tsar_dump_path)
        filenames = tar.getnames()
        tar.extractall(TMP_DIR_NAME)
        tar.close()
        logging.info('Finish unpacking file')

        if set(filenames) == {'st_model_dump'}:
            st_tsar_dump_path = os.path.join(TMP_DIR_NAME, "st_model_dump")
        else:
            logging.info('Converting to sthash')
            to_sthash_binary = get_last_resource(to_sthash_converter_params)

            st_tsar_dump_dir = os.path.join(TMP_DIR_NAME, "result")
            st_tsar_dump_path = os.path.join(st_tsar_dump_dir, "st_model_dump")
            sp.check_call(
                "cat {tmp_dir}/* > {tmp_dir}/normal_model_dump".format(tmp_dir=TMP_DIR_NAME),
                shell=True
            )
            sp.check_call(
                "mkdir {st_dump_dir}".format(st_dump_dir=st_tsar_dump_dir),
                shell=True
            )
            sp.check_call(
                "{to_sthash_binary} --input_file {tmp_dir}/normal_model_dump --output_file {st_dump_path} --compression_type {compression_type} --format {model_format}".format(
                    tmp_dir=TMP_DIR_NAME,
                    to_sthash_binary=to_sthash_binary,
                    st_dump_path=st_tsar_dump_path,
                    compression_type=compression_type,
                    model_format=model_format
                ),
                shell=True
            )
            logging.info('Finish converting to sthash')

    logging.info('Cleaning tmp dir')
    shutil.move(st_tsar_dump_path, dump_dst)
    shutil.rmtree(TMP_DIR_NAME)
    logging.info('Finish cleaning tmp dir')


def make_difacto_dump(task_id, compression_type='float', model_format='sthash'):
    dump_params = {
        "type": "TSAR_DUMP",
        "state": "READY",
        "attrs": {"task_id": task_id},
    }

    dump_dst = task_id + 'st_model_dump'

    resource = get_last_resource_by_params(dump_params)

    convert_resource_to_sthash_dump(resource, dump_dst, compression_type, model_format)

    return dump_dst
