from logging import info
import os
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.common.types.resource import State

from sandbox.projects.ads.watchman_plotter.resources import WatchmanPlotState, WatchmanPlotterBinary


def get_last_log_date(graph_name):
    resource = WatchmanPlotState.find(
        attrs={"graph_name": graph_name},
        state=(State.READY,),
    ).order(-sdk2.Resource.id).first()
    if resource:
        return resource.last_log_date


class PlotWatchmanGraph(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        yt_token_owner = sdk2.parameters.String(
            "YT token owner",
            required=True
        )

        yt_token_name = sdk2.parameters.String(
            "YT token name",
            required=True
        )

        plotter_config = sdk2.parameters.String(
            "Plotter config",
            required=True,
            multiline=True
        )

        start_time = sdk2.parameters.String(
            "Start time",
            required=True
        )

        graph_name = sdk2.parameters.String(
            "Graph name",
            required=True
        )

    def get_env_with_yt_token(self):
        yt_token = sdk2.Vault.data(self.Parameters.yt_token_owner, self.Parameters.yt_token_name)
        env = os.environ.copy()
        env['YT_TOKEN'] = yt_token
        return env

    def on_execute(self):
        binary = sdk2.Resource.find(resource_type=WatchmanPlotterBinary,
                                    attrs={"released": "stable"}).order(-sdk2.Resource.id).first()
        binary_path = str(sdk2.ResourceData(binary).path)

        info('Started parsing conf')

        start_time = get_last_log_date(self.Parameters.graph_name)
        if start_time is None:
            start_time = self.Parameters.start_time
        else:
            date_fmt = "%Y-%m-%d-%H"
            start_time = (datetime.strptime(start_time, date_fmt) + timedelta(hours=1)).strftime(date_fmt)
        if start_time is None:
            raise ValueError("No start time!")

        plotter_conf = self.Parameters.plotter_config
        conf_file = 'graph.conf'
        with open(conf_file, 'w') as f:
            f.write(plotter_conf)

        cmd = [
            binary_path,
            "--conf", conf_file,
            "--start_date", start_time
        ]

        info('Running command {}'.format(cmd))

        with sdk2.helpers.ProcessLog(self, logger='compute_metrics') as pl:
            sdk2.helpers.subprocess.check_call(
                cmd,
                env=self.get_env_with_yt_token(),
                stdout=pl.stdout,
                stderr=pl.stderr
            )

        with open('last_date.txt') as f:
            last_log_date = f.read()

        state = sdk2.ResourceData(
            WatchmanPlotState(
                self,
                "State for computing offline metrics",
                path='last_date.txt',
                last_log_date=last_log_date,
                graph_name=self.Parameters.graph_name
            )
        )
        state.ready()
