import os
import json
import logging

from sandbox import sdk2
from sandbox.common.types.resource import State
from sandbox.projects.ads.watchman_scheduler.resources import WatchmanSchedulerState, WatchmanSchedulerBinary
import sandbox.common.types.notification as ctn


def get_last_state(state_key):
    resource = WatchmanSchedulerState.find(
        attrs={"state_key": state_key},
        state=(State.READY,),
    ).order(-sdk2.Resource.id).first()
    if resource:
        return resource


class ScheduleWatchmanGraph(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        yt_token_owner = sdk2.parameters.String(
            "YT token owner",
            required=True
        )

        yt_token_name = sdk2.parameters.String(
            "YT token name",
            required=True
        )

        nirvana_token_owner = sdk2.parameters.String(
            "Nirvana token owner",
            required=True
        )

        nirvana_token_name = sdk2.parameters.String(
            "Nirvana token name",
            required=True
        )

        scheduler_config = sdk2.parameters.String(
            "Scheduler configuration",
            required=True,
            multiline=True
        )

        reporter_configuration = sdk2.parameters.String(
            "Reporter configuration",
            required=True,
            multiline=True
        )

        notify_on_failure = sdk2.parameters.List(
            'Staff logins',
            sdk2.parameters.String
        )

        task_id = sdk2.parameters.String(
            "Task id",
            required=True
        )

        task_configuration = sdk2.parameters.String(
            "Task configuration",
            required=True,
            multiline=True
        )

        scheduler_name = sdk2.parameters.String(
            "Scheduler state key",
            required=True
        )

        scheduler_binary = sdk2.parameters.Resource(
            'Scheduler binary',
            required=True,
            resource_type=WatchmanSchedulerBinary,
            state=(State.READY,),
        )

    def on_execute(self):
        binary_path = str(sdk2.ResourceData(self.Parameters.scheduler_binary).path)

        state_file = 'state.json'
        report_file = 'report.txt'

        cmd = [
            binary_path,
            "--scheduler_conf", json.dumps(json.loads(self.Parameters.scheduler_config)),
            "--reporter_conf", json.dumps(json.loads(self.Parameters.reporter_configuration)),
            "--task_id", self.Parameters.task_id,
            "--task_conf", json.dumps(json.loads(self.Parameters.task_configuration)),
            "--state_output_file", state_file,
            "--report_file", report_file
        ]

        state = get_last_state(self.Parameters.scheduler_name)

        if state is not None:
            cmd += ["--state_input_file", str(sdk2.ResourceData(state).path)]

        env = os.environ.copy()
        env['YT_TOKEN'] = sdk2.Vault.data(self.Parameters.yt_token_owner, self.Parameters.yt_token_name)
        env['NIRVANA_TOKEN'] = sdk2.Vault.data(self.Parameters.nirvana_token_owner, self.Parameters.nirvana_token_name)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(__name__)) as pl:
            sdk2.helpers.subprocess.check_call(
                cmd,
                stdout=pl.stdout,
                stderr=pl.stderr,
                env=env
            )

        state = sdk2.ResourceData(
            WatchmanSchedulerState(
                self,
                "Watchman scheduler state",
                path=state_file,
                state_key=self.Parameters.scheduler_name
            )
        )
        state.ready()

        if os.path.isfile(report_file):
            with open(report_file) as f:
                msg = f.read()
                self.server.notification(
                    subject="Scheduler {} fails!".format(self.Parameters.scheduler_name),
                    body=msg.encode('utf-8'),
                    recipients=self.Parameters.notify_on_failure,
                    transport=ctn.Transport.EMAIL
                )
