# -*- coding: utf-8 -*-

import os
import logging
from sandbox import sdk2

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.parameters import SandboxBoolGroupParameter

from sandbox.projects.common.build import parameters as ya_make_parameters
from sandbox.projects.common.build.YaMake import YaMakeTask

from sandbox.projects.common.nanny import nanny
from sandbox.projects.adv_machine.common import parameters, resources, send_telegram_notification


logger = logging.getLogger(__name__)


ResourceDecs = {
    'config': resources.AdvMachineConfig,
    'runtime_bins': resources.AdvMachineRuntimeBin,
    'indexer_bins': resources.AdvMachineIndexerBin,
    'indexer_config': resources.AdvMachineIndexerConfig,
    'worker_tools_bins': resources.AdvMachineWorkerTools,
    'superapp_http_direct': resources.SuperappHttpDirectBin,
}


def is_config(resource_name):
    return resource_name == 'config'


def get_resource_path(resource_name):
    if is_config(resource_name):
        resource_path = ResourceDecs[resource_name].arts_source[0]
        return os.path.basename(os.path.dirname(resource_path))
    return resource_name


def get_release_services():
    release_services = set()
    for resource in ResourceDecs.values():
        release_services.update(resource.release_services)
    release_services = list(release_services)
    release_services.sort()
    return release_services


class Resources(SandboxBoolGroupParameter):
    name = 'resources'
    required = False
    description = 'Resources to build'
    choices = [(v.description, k) for k, v in ResourceDecs.iteritems()]


class RuntimeReleaseServicesParameter(SandboxBoolGroupParameter):
    name = 'runtime_release_services'
    description = 'Nanny services to release'
    choices = [(s, s) for s in get_release_services()]
    group = parameters.RELEASE_PARAMETERS_GROUP_NAME


class BuildAdvMachineParameters(sdk2.Parameters):
    checkout_arcadia_from_url = ya_make_parameters.ArcadiaUrl()
    arcadia_patch = ya_make_parameters.ArcadiaPatch()
    build_system = ya_make_parameters.BuildSystem()
    build_type = ya_make_parameters.BuildType()
    sanitize = ya_make_parameters.Sanitize()
    strip_binaries = ya_make_parameters.StripBinaries()
    use_aapi_fuse = ya_make_parameters.UseArcadiaApiFuse(default_value=True)
    ya_yt_store = ya_make_parameters.YtStore(default_value=True)
    ya_yt_token_vault_name = ya_make_parameters.YtTokenVaultName(default_value='adv_machine_yt_token')
    lto = ya_make_parameters.LTO()


class BuildAdvMachine(YaMakeTask):

    """ Attention, please use BUILD_ADV_MACHINE_2 """

    type = 'BUILD_ADV_MACHINE'
    execution_space = 200 * 1024
    cores = 50

    environment = (environments.PipEnvironment('startrek_client'),)

    input_parameters = list(BuildAdvMachineParameters) + [
        Resources,
        RuntimeReleaseServicesParameter,
        parameters.DeployAfterRelease,
        parameters.DeployTicketIdParameter,
    ]

    client_tags = YaMakeTask.client_tags & ctc.Tag.Group.LINUX

    def get_resources(self):
        resources = {}
        for resource_name, resource in self._chosen_resources():
            resources[resource_name] = {
                'description': '{}: {}'.format(resource.description, self.descr),
                'resource_type': resource,
                'resource_path': get_resource_path(resource_name),
            }
        return resources

    def get_targets(self):
        targets = []
        for resource_name, resource in self._chosen_resources():
            targets += [os.path.dirname(art) for art in resource.arts]
        return targets

    def get_arts(self):
        arts = []
        for resource_name, resource in self._chosen_resources():
            arts += [{'path': art, 'dest': get_resource_path(resource_name)} for art in resource.arts]
        return arts

    def get_arts_source(self):
        arts_source = []
        for resource_name, resource in self._chosen_resources():
            arts_source += [{'path': art, 'dest': get_resource_path(resource_name)} for art in resource.arts_source]
        return arts_source

    def on_execute(self):
        YaMakeTask.on_execute(self)

        release_services = self.ctx[RuntimeReleaseServicesParameter.name] or ''
        release_services = release_services.strip().split()
        resources = [_ for _ in self.list_resources() if _.type.releasable]
        self._send_startrek_notification('Resource {} will be released to {}'.format(
            ', '.join([str(r.type) for r in resources]),
            ', '.join(release_services)
        ))

    def on_release(self, additional_parameters):
        YaMakeTask.on_release(self, additional_parameters)

        if additional_parameters['release_status'].lower() != 'stable':
            self.set_info('Resources were released to {} status'.format(additional_parameters['release_status']))
            return

        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('ADV_MACHINE', 'adv_machine_nanny_token')
        )

        release_services = self.ctx[RuntimeReleaseServicesParameter.name].strip().split()
        for service_id in release_services:
            nanny_client.update_service_sandbox_file(
                service_id=service_id,
                task_type=self.type,
                task_id=str(self.id),
                deploy=self.ctx[parameters.DeployAfterRelease.name],
                skip_not_existing_resources=True,
                allow_empty_changes=True,
            )

        resources = [_ for _ in self.list_resources() if _.type.releasable]
        comment = 'Resource {} was released to {}'.format(
            ', '.join([str(r.type) for r in resources]),
            ', '.join(release_services)
        )

        self.set_info(comment)
        send_telegram_notification(
            'Новые {} с низким бидом покатим на {}. Благодарить {}. Таск {}'.format(
                ', '.join([str(r.type) for r in resources]),
                ', '.join(release_services),
                additional_parameters['releaser'],
                self.http_url(),
            ),
            self.get_vault_data('ADV_MACHINE', 'adv_machine_telegram_bot_token')
        )

    def _chosen_resources(self):
        resource_names = self.ctx.get(Resources.name).strip().split()
        return ((name, ResourceDecs[name]) for name in resource_names)

    def _send_startrek_notification(self, comment):
        ticket_id = self.ctx[parameters.DeployTicketIdParameter.name]
        if not ticket_id:
            return

        from startrek_client import Startrek
        st_client = Startrek(token=self.get_vault_data('adv_machine_st_token'), useragent='robot-adv-machine')
        issue = st_client.issues[ticket_id]
        issue.comments.create(text='{}. Author {}. Task {}'.format(comment, self.author, self.http_url()))


__Task__ = BuildAdvMachine
