# -*- coding: utf-8 -*-

import logging

from datetime import datetime, timedelta

from sandbox import sdk2
import sandbox.sandboxsdk.environments as sdk_environments

from sandbox.projects.adv_machine.common import get_yt_config, ConfigFormat
from sandbox.projects.adv_machine.common.parameters import YTParameters


logger = logging.getLogger(__name__)


class CleanYtAdvqualityQuota(sdk2.Task):

    """Regular task to clean yt quota for advquality"""

    class Requirements(sdk2.Requirements):
        cores = 1
        environments = [sdk_environments.PipEnvironment('yandex-yt'), ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('YT nodes') as nodes_block:
            nodes = sdk2.parameters.Dict('Node names and storage period in days', required=True)

        yt = YTParameters

    def on_execute(self):
        from yt.wrapper import YtClient, create_client_with_command_params

        yt = YtClient(config=get_yt_config(self.Parameters.yt, self.author, ConfigFormat.CLIENT))
        yt = create_client_with_command_params(yt, **{"suppress_access_tracking": True})

        self.removed_nodes = []
        for node, period in self.Parameters.nodes.iteritems():
            end_date = datetime.today() - timedelta(days=int(period))
            self._clean_node(yt, node, end_date)

        if self.removed_nodes:
            self.set_info('Remove:\n{}'.format(
                '\n'.join(
                    ['{}\t{}'.format(n, n.attributes['access_time'].split('T')[0]) for n in self.removed_nodes]
                )
            ))

    def _clean_node(self, yt, root_node, end_date):
        is_empty_directory = True

        for node in sorted(yt.list(root_node, attributes=['type', 'access_time', 'broken'], absolute=True)):
            time = _convert_to_datetime(node.attributes.get('access_time'))

            if node.attributes['type'] == 'map_node':
                is_empty_directory = False

                if self._clean_node(yt, node, end_date) and time < end_date:
                    # empty child created long ago
                    yt.remove(node)
                    self.removed_nodes.append(node)
                continue

            node_type = node.attributes['type']
            if node_type != 'link' and node_type != 'table':
                is_empty_directory = False
                continue
            if node_type == 'link' and not node.attributes['broken']:
                is_empty_directory = False
                continue
            if node_type == 'table' and time >= end_date:
                is_empty_directory = False
                continue

            yt.remove(node)
            self.removed_nodes.append(node)

        return is_empty_directory


def _convert_to_datetime(yt_time):
    return datetime.strptime(yt_time, '%Y-%m-%dT%H:%M:%S.%fZ')
