# -*- coding: utf-8 -*-

import os
from sandbox import sdk2

import sandbox.common.types.client as ctc

from sandbox.projects.common.build import parameters as ya_make_parameters
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.nanny import nanny
from sandbox.projects.adv_machine.common import parameters, send_telegram_notification
from sandbox.sandboxsdk.parameters import SandboxBoolGroupParameter


class BuildAdvMachineBaseParameters(sdk2.Parameters):
    checkout_arcadia_from_url = ya_make_parameters.ArcadiaUrl()
    arcadia_patch = ya_make_parameters.ArcadiaPatch()
    build_system = ya_make_parameters.BuildSystem()
    build_type = ya_make_parameters.BuildType()
    sanitize = ya_make_parameters.Sanitize()
    strip_binaries = ya_make_parameters.StripBinaries()
    use_aapi_fuse = ya_make_parameters.UseArcadiaApiFuse(default_value=True)
    ya_yt_store = ya_make_parameters.YtStore(default_value=True)
    ya_yt_token_vault_name = ya_make_parameters.YtTokenVaultName(default_value='adv_machine_yt_token')
    lto = ya_make_parameters.LTO()


class BuildAdvMachineBase(nanny.ReleaseToNannyTask, YaMakeTask):
    execution_space = 300 * 1024
    cores = 50

    input_parameters = list(BuildAdvMachineBaseParameters)

    def on_execute(self):
        self.ctx['ya_yt_token_vault_name'] = 'adv_machine_yt_token'
        YaMakeTask.on_execute(self)


class BuildAdvMachineRobotParametersFactory(object):
    @staticmethod
    def generate_items_to_build(resource_decs):
        class ItemsToBuild(SandboxBoolGroupParameter):
            name = 'items_to_build'
            required = False
            description = 'Build'
            choices = [(v['description'], k) for k, v in resource_decs.iteritems()]
        return ItemsToBuild

    @staticmethod
    def get_release_services(resource_decs):
        release_services = set()
        for resource in resource_decs.values():
            release_services.update(resource['type'].release_services)
        return release_services

    @classmethod
    def generate_release_services(cls, resource_decs):
        class ReleaseServicesParameter(SandboxBoolGroupParameter):
            name = 'runtime_release_services'
            description = 'Nanny services to release'
            choices = [(s, s) for s in cls.get_release_services(resource_decs)]
            group = parameters.RELEASE_PARAMETERS_GROUP_NAME
        return ReleaseServicesParameter


class BuildAdvMachineRobotBase(BuildAdvMachineBase):

    """
        Base class for robot
        Override ReleaseServicesParameter, ResourceDecs, ItemsToBuild
    """

    ReleaseServicesParameter = None
    ItemsToBuild = None
    ResourceDecs = {}

    client_tags = ctc.Tag.Group.LINUX & YaMakeTask.client_tags

    def get_resources(self):
        resources = {}
        for resource_name, resource in self._chosen_resources():
            resources[resource_name] = {
                'description': '{}: {}'.format(resource.description, self.descr),
                'resource_type': resource,
                'resource_path': resource_name,
            }
        return resources

    def get_targets(self):
        targets = []
        for resource_name, resource in self._chosen_resources():
            targets += [os.path.dirname(art) for art in resource.arts]
            targets += [art for art in resource.arts_union]
        return targets

    def get_arts(self):
        arts = []
        for resource_name, resource in self._chosen_resources():
            arts += [{'path': art, 'dest': resource_name} for art in resource.arts]
            arts += [{'path': art, 'dest': resource_name} for art in resource.arts_union]
        return arts

    def get_arts_source(self):
        arts_source = []
        for resource_name, resource in self._chosen_resources():
            arts_source += [{'path': art, 'dest': resource_name} for art in resource.arts_source]
        return arts_source

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

        resources = [_ for _ in self.list_resources() if _.type.releasable]
        send_telegram_notification(
            '{} выехал. Инициировал {}. Sad box url: {}'.format(
                ', '.join([str(r.type) for r in resources]),
                additional_parameters['releaser'],
                self.http_url(),
            ),
            self.get_vault_data('ADV_MACHINE', 'adv_machine_telegram_bot_token')
        )

        YaMakeTask.on_release(self, additional_parameters)

    def _chosen_resources(self):
        resource_names = self.ctx.get(self.ItemsToBuild.name).strip().split()
        return ((name, self.ResourceDecs[name]['type']) for name in resource_names)
