# -*- coding: utf-8 -*-

import collections

from sandbox import sdk2
from sandbox.common import telegram
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import sandbox.sandboxsdk.environments as sdk_environments

from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.nanny import const

telegram_chat_id = -147436600


def send_telegram_notification(message, token):
    bot = telegram.TelegramBot(token)
    bot.send_message(telegram_chat_id, message)


class AdvMachineReleaseTask(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        environments = [sdk_environments.PipEnvironment('startrek_client', version='2.3.0', custom_parameters=['--upgrade-strategy only-if-needed']), ]

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Release parameters') as release_block:
            with sdk2.parameters.CheckGroup('Nanny groups to auto release', default=[]) as nanny_labels:
                groups = [
                    'default', 'rsya', 'search', 'robot',
                    'quorums-machine', 'quorums-machine-meta', 'quorums-machine-daemons',
                    'turbo', 'edadeal', 'edadeal_robot', 'offer_base', 'offer_base_robot',
                    'exp', 'exp_robot', 'games'
                ]
                nanny_labels.choices = [(s, s) for s in groups]

            release_labels = nanny.LabelsParameter2(
                'Release Labels',
                description=u'Метки, которые будут добавлены к релизу (для фильтрации в Няне)',
            )

            with sdk2.parameters.CheckGroup('Nanny services to release', default=[], ui=None) as release_services:
                release_services.choices = []
            additional_release_services = sdk2.parameters.String(
                'Additional comma separated list of Nanny services to release', ui=None)

            deploy_after_release = sdk2.parameters.Bool('Deploy after release automatically', default=True, ui=None)
            create_deploy_ticket = sdk2.parameters.Bool('Create deploy startrek ticket')
            with create_deploy_ticket.value[False]:
                deploy_ticket_id = sdk2.parameters.String('Deploy startrek ticket id')
            send_telegram_message = sdk2.parameters.Bool('Send telegram message', default=True)

    def on_execute(self):
        resources = filter(lambda _: _.releasable,
                           sdk2.Resource.find(task=self, state=(ctr.State.READY, ctr.State.NOT_READY)).limit(0))
        release_services = self._get_release_services()

        st_message = self._get_st_message(resources, release_services)
        if self.Parameters.create_deploy_ticket:
            self._create_ticket(description=st_message)
        else:
            self._send_startrek_notification(comment=st_message)

        deploy_ticket_id = self.Parameters.deploy_ticket_id or self.Context.ticket_id
        if deploy_ticket_id:
            self.Parameters.tags += [deploy_ticket_id]

    def get_release_labels(self):
        labels = self.Parameters.release_labels
        labels['groups'] = ','.join(self.Parameters.nanny_labels)
        return labels

    def on_release(self, params):
        sdk2.Task.on_release(self, params)

        nanny_client = NannyClient(
            api_url=const.NANNY_API_URL,
            oauth_token=sdk2.Vault.data('adv_machine_nanny_token'),
        )

        data = collections.defaultdict(dict, self.get_nanny_release_info(params))
        data['meta']['labels'] = self.get_release_labels()
        result = nanny_client.create_release2(dict(data))['value']
        release_id = result['id']
        if self.Context.nanny_release_requests is ctm.NotExists:
            self.Context.nanny_release_requests = []
        self.Context.nanny_release_requests.append(release_id)
        release_link = const.RELEASE_REQUEST_TEMPLATE.format(
            nanny_api_url=const.NANNY_API_URL,
            release_request_id=release_id,
        )
        self.set_info(
            'Nanny release request <a href="{}">{}</a> was created.'.format(release_link, release_id),
            do_escape=False
        )

    def _get_st_message(self, resources, release_services):
        return self.Parameters.description

    def _get_telegram_message(self, params, resources, release_services):
        return params['release_comments'] or self.Parameters.description

    def _get_release_services(self):
        additional_release_services = self.Parameters.additional_release_services.split(',')
        additional_release_services = [_.strip() for _ in additional_release_services if _]
        return set(self.Parameters.release_services + additional_release_services)

    def _send_startrek_notification(self, comment):
        if not self.Parameters.deploy_ticket_id:
            return

        from startrek_client import Startrek
        st_client = Startrek(token=sdk2.Vault.data('adv_machine_st_token'), useragent='robot-adv-machine')
        issue = st_client.issues[self.Parameters.deploy_ticket_id]
        issue.comments.create(text=comment)

    def _create_ticket(self, description):
        from startrek_client import Startrek
        st_client = Startrek(token=sdk2.Vault.data('adv_machine_st_token'), useragent='robot-adv-machine')
        ticket = st_client.issues.create(
            queue='AMREL',
            summary='{}'.format(self.Parameters.description),
            description=description,
            type='task',
            followers=[self.author],
        )
        self.Context.ticket_id = ticket.key
        self.set_info(
            'Created <a href="https://st.yandex-team.ru/{issue_id}">{issue_id}</a>'.format(issue_id=ticket.key),
            do_escape=False)


class AdvMachineReleaseTaskServiceName(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        environments = [sdk_environments.PipEnvironment('startrek_client', version='2.3.0', custom_parameters=['--upgrade-strategy only-if-needed']), ]

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Required parameters') as release_block:
            with sdk2.parameters.CheckGroup('Services', default=[]) as nanny_labels:
                groups = [
                    "adv-machine-rsya-quorums-machine",
                    "adv-machine-offer-retargeting-big-base",
                    "adv-machine-rsya-rm-machine",
                    "adv-machine-synonyms-apphost",
                    "adv-machine-monoindex-apphost",
                    "adv-machine-rsya-rm-autobudget",
                    "adv-machine-rsya-retargeting",
                    "adv-machine-rsya-rmp",
                    "adv-machine-all-dsa-10-shards-apphost",
                    "adv-machine-search-query-recommender",
                    "adv-machine-1cp",
                    "adv-machine-edadeal",
                    "adv-machine-games",
                ]
                nanny_labels.choices = [(s, s) for s in groups]

        with sdk2.parameters.Group('Release parameters') as release_block:

            release_labels = nanny.LabelsParameter2(
                'Release Labels',
                description=u'Метки, которые будут добавлены к релизу (для фильтрации в Няне)',
            )

            with sdk2.parameters.CheckGroup('Nanny services to release', default=[], ui=None) as release_services:
                release_services.choices = []
            additional_release_services = sdk2.parameters.String(
                'Additional comma separated list of Nanny services to release', ui=None)

            deploy_after_release = sdk2.parameters.Bool('Deploy after release automatically', default=True, ui=None)
            create_deploy_ticket = sdk2.parameters.Bool('Create deploy startrek ticket')
            with create_deploy_ticket.value[False]:
                deploy_ticket_id = sdk2.parameters.String('Deploy startrek ticket id')
            send_telegram_message = sdk2.parameters.Bool('Send telegram message', default=True)

    def on_execute(self):
        resources = filter(lambda _: _.releasable,
                           sdk2.Resource.find(task=self, state=(ctr.State.READY, ctr.State.NOT_READY)).limit(0))
        release_services = self._get_release_services()

        st_message = self._get_st_message(resources, release_services)
        if self.Parameters.create_deploy_ticket:
            self._create_ticket(description=st_message)
        else:
            self._send_startrek_notification(comment=st_message)

        deploy_ticket_id = self.Parameters.deploy_ticket_id or self.Context.ticket_id
        if deploy_ticket_id:
            self.Parameters.tags += [deploy_ticket_id]

    def get_release_labels(self):
        labels = self.Parameters.release_labels
        labels['groups'] = ','.join(self.Parameters.nanny_labels)
        return labels

    def get_services(self):
        return self.Parameters.nanny_labels

    def on_release(self, params):
        sdk2.Task.on_release(self, params)

        nanny_client = NannyClient(
            api_url=const.NANNY_API_URL,
            oauth_token=sdk2.Vault.data('adv_machine_nanny_token'),
        )

        data = collections.defaultdict(dict, self.get_nanny_release_info(params))
        data['meta']['labels'] = self.get_release_labels()
        result = nanny_client.create_release2(dict(data))['value']
        release_id = result['id']
        release_link = const.RELEASE_REQUEST_TEMPLATE.format(
            nanny_api_url=const.NANNY_API_URL,
            release_request_id=release_id,
        )
        self.set_info(
            'Nanny release request <a href="{}">{}</a> was created.'.format(release_link, release_id),
            do_escape=False
        )

    def _get_st_message(self, resources, release_services):
        return self.Parameters.description

    def _get_telegram_message(self, params, resources, release_services):
        return params['release_comments'] or self.Parameters.description

    def _get_release_services(self):
        additional_release_services = self.Parameters.additional_release_services.split(',')
        additional_release_services = [_.strip() for _ in additional_release_services if _]
        return set(self.Parameters.release_services + additional_release_services)

    def _send_startrek_notification(self, comment):
        if not self.Parameters.deploy_ticket_id:
            return

        from startrek_client import Startrek
        st_client = Startrek(token=sdk2.Vault.data('adv_machine_st_token'), useragent='robot-adv-machine')
        issue = st_client.issues[self.Parameters.deploy_ticket_id]
        issue.comments.create(text=comment)

    def _create_ticket(self, description):
        from startrek_client import Startrek
        st_client = Startrek(token=sdk2.Vault.data('adv_machine_st_token'), useragent='robot-adv-machine')
        ticket = st_client.issues.create(
            queue='AMREL',
            summary='{}'.format(self.Parameters.description),
            description=description,
            type='task',
            followers=[self.author],
        )
        self.Context.ticket_id = ticket.key
        self.set_info(
            'Created <a href="https://st.yandex-team.ru/{issue_id}">{issue_id}</a>'.format(issue_id=ticket.key),
            do_escape=False)
