# -*- coding: utf-8 -*-

from enum import Enum

from sandbox import sdk2


adv_machine_robots = ['robot-adv-machine', 'robot-am-indexer', 'robot-am-helper']


class ConfigFormat(Enum):
    ENVIRONMENT = 1
    CLIENT = 2


def _detect_yt_pool(yt_params, author=None):
    if yt_params.pool:
        return yt_params.pool
    elif author in adv_machine_robots:
        return 'adv-machine'
    return None


def _get_prefix_key(format=ConfigFormat.ENVIRONMENT):
    return {
        ConfigFormat.CLIENT: 'prefix',
        ConfigFormat.ENVIRONMENT: 'YT_PREFIX'
    }[format]


def _update_prefix(config, prefix, format=ConfigFormat.ENVIRONMENT):
    config.update({_get_prefix_key(format): prefix})


def _get_token_key(format=ConfigFormat.ENVIRONMENT):
    return {
        ConfigFormat.CLIENT: 'token',
        ConfigFormat.ENVIRONMENT: 'YT_TOKEN'
    }[format]


def _update_token(config, token_vault, format=ConfigFormat.ENVIRONMENT):
    config.update({_get_token_key(format): sdk2.Vault.data(token_vault)})


def _get_pool_key(format=ConfigFormat.ENVIRONMENT):
    return {
        ConfigFormat.CLIENT: 'pool',
        ConfigFormat.ENVIRONMENT: 'YT_POOL'
    }[format]


def _update_pool(config, pool, format=ConfigFormat.ENVIRONMENT):
    config.update({_get_pool_key(format): pool})


def _update_proxy(config, proxy, format=ConfigFormat.ENVIRONMENT):
    if format == ConfigFormat.ENVIRONMENT:
        config['YT_PROXY'] = proxy
    else:
        config['proxy'] = {'url': proxy}


def get_yt_config(yt_params, author=None, format=ConfigFormat.ENVIRONMENT):
    config = {}

    proxy = yt_params.proxy
    if proxy:
        _update_proxy(config, proxy, format)

    prefix = yt_params.prefix
    if prefix:
        _update_prefix(config, prefix, format)

    token_vault = yt_params.token_vault
    if token_vault:
        _update_token(config, token_vault, format)

    pool = _detect_yt_pool(yt_params, author)
    if pool:
        _update_pool(config, pool, format)

    return config
