# -*- coding: utf-8 -*-

import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.sandboxsdk.environments as sdk_environments

import sandbox.common.types.resource as ctr

from sandbox.projects.adv_machine.common import process_wrapper
from sandbox.projects.adv_machine.common import unpack_tables_to_yt
from sandbox.projects.adv_machine.common.parameters import YTLocalParameters
from sandbox.projects.adv_machine.common.yt_local import YtLocal

from sandbox.projects.adv_machine.common.resources import AdvMachineCompareIndexBin
from sandbox.projects.adv_machine.common.resources import AdvMachinePreIndexTable
from sandbox.projects.adv_machine.common.resources import AdvMachineYtLocalLogs


class AdvMachineSplittedIndexDiff(sdk2.Resource):
    releasable = False
    any_arch = True
    auto_backup = True


class CompareAdvMachineSplittedIndex(sdk2.Task):

    """Compares AdvMachine splitted index tables"""

    class Requirements(sdk2.Task.Requirements):
        ram = 100 * 1024  # Why so much?
        environments = [sdk_environments.PipEnvironment('yandex-yt'), sdk_environments.PipEnvironment('yandex-yt-yson-bindings-skynet'), ]

    class Parameters(sdk2.Task.Parameters):
        yt = YTLocalParameters

        with sdk2.parameters.Group('Splitted tables') as pre_index_tables_block:
            original_tables = sdk2.parameters.Resource(
                'Original tables',
                resource_type=AdvMachinePreIndexTable,
                state=(ctr.State.READY, ),
                required=True,
            )
            new_tables = sdk2.parameters.Resource(
                'New tables',
                resource_type=AdvMachinePreIndexTable,
                state=(ctr.State.READY, ),
                required=True,
            )

        with sdk2.parameters.Group('Executable parameters') as binaries_block:
            compare_bin = sdk2.parameters.LastReleasedResource(
                'Compare index tables binaries',
                resource_type=AdvMachineCompareIndexBin,
                state=(ctr.State.READY, ),
                required=True,
            )

    class Context(sdk2.Task.Context):
        has_diff = False
        metrics_diff_json = None

    def on_execute(self):
        import yt.wrapper as yt

        original_tables = sdk2.ResourceData(self.Parameters.original_tables)
        new_tables = sdk2.ResourceData(self.Parameters.new_tables)
        compare_binary = sdk2.ResourceData(self.Parameters.compare_bin)

        result_diff_folder = 'diff'
        if not os.path.exists(result_diff_folder):
            os.makedirs(result_diff_folder)
        assert not os.listdir(result_diff_folder)  # should be empty

        yt_local_memory_limit = 10 << 30

        with YtLocal(yt_local_res=self.Parameters.yt.yt_local, ncpu=5, memory=yt_local_memory_limit, node_chunk_store_quota=yt_local_memory_limit) as yt_local:
            proxy_url = yt_local.yt_runner.get_proxy_string()
            yt_prefix = self.Parameters.yt.prefix
            yt_client = CompareAdvMachineSplittedIndex._create_yt_client(proxy_url, yt_prefix)

            original_tables_prefix = yt.ypath_join(yt_prefix, 'original_tables')
            new_tables_prefix = yt.ypath_join(yt_prefix, 'new_tables')

            try:
                unpack_tables_to_yt(yt_client, original_tables_prefix, original_tables.path, with_attrs=True)
                unpack_tables_to_yt(yt_client, new_tables_prefix, new_tables.path, with_attrs=True)

                cmd = [
                    str(compare_binary.path),
                    '-s', proxy_url,
                    '--i1', original_tables_prefix,
                    '--i2', new_tables_prefix,
                    '-o', result_diff_folder,
                ]

                with process_wrapper(self, logger='compare_index_tables') as pl:
                    sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr)

                if os.listdir(result_diff_folder):  # there is some diff
                    AdvMachineSplittedIndexDiff(self, self.Parameters.description, result_diff_folder)
                    self.Context.has_diff = True
                else:
                    self.Context.has_diff = False

            except Exception:
                AdvMachineYtLocalLogs(self, 'AdvMachine yt local logs: {}'.format(self.Parameters.description), yt_local.yt_log_path)
                raise
            else:
                AdvMachineYtLocalLogs(self, 'AdvMachine yt local logs: {}'.format(self.Parameters.description), yt_local.yt_log_path)

    @staticmethod
    def _create_yt_client(yt_proxy, yt_prefix):
        import yt.wrapper as yt

        yt_client = yt.YtClient(config=yt.config.config)
        if yt_proxy:
            yt_client.config['proxy']['url'] = yt_proxy
        if yt_prefix:
            yt_client.config['prefix'] = yt_prefix

        return yt_client
