# -*- coding: utf-8 -*-
import logging
import uuid

from os.path import join
from sandbox import sdk2
from sandbox import common

import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

from sandbox.projects.adv_machine.common import get_yt_config, process_wrapper, resources
from sandbox.projects.adv_machine.common.parameters import YTParameters
from sandbox.projects.common import binary_task
from sandbox.sdk2.helpers import subprocess as sp


logger = logging.getLogger(__name__)


class ShellAdvMachineRuntimeExp(binary_task.LastBinaryTaskRelease, sdk2.Task):

    """Задача для обстрела меты сервиса патронами с подмерженными экспериментами"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        disk_space = 10 * 1024

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Input resources') as input_res_block:
            runtime_bins = sdk2.parameters.LastReleasedResource(
                'Priemka bin',
                resource_type=resources.AdvMachineRuntimeBin,
                state=(ctr.State.READY, ),
                required=True,
            )
            worker_tools = sdk2.parameters.LastReleasedResource(
                'Ammo prepare bin',
                resource_type=resources.AdvMachineWorkerTools,
                state=(ctr.State.READY, ),
                required=True,
            )

        with sdk2.parameters.Group('Required parameters') as parameters:
            with sdk2.parameters.CheckGroup('Services') as service_names:
                service_names.values["adv-machine-offer-retargeting-big-base-prestable-meta"] = service_names.Value(value="adv-machine-offer-retargeting-big-base-prestable-meta")
                service_names.values["adv-machine-rsya-rm-machine-prestable-yp"] = service_names.Value(value="adv-machine-rsya-rm-machine-prestable-yp")
                service_names.values["adv-machine-rsya-rmp-prestable-yp"] = service_names.Value(value="adv-machine-rsya-rmp-prestable-yp")
                service_names.values["adv-machine-search-synonyms-testing-bin-meta"] = service_names.Value(value="adv-machine-search-synonyms-testing-bin-meta")
                service_names.values["adv-machine-rsya-retargeting-prestable-yp"] = service_names.Value(value="adv-machine-rsya-retargeting-prestable-yp")

            ab_config = sdk2.parameters.String('AB experiment config', multiline=True, required=True)

        with sdk2.parameters.Group('Optional parameters') as optional_param_block:
            ammo_format = sdk2.parameters.String(
                'Ammo format from here: https://a.yandex-team.ru/arc/trunk/arcadia/ads/quality/adv_machine/lib/priemka/request/parse/parse.h?blame=true&rev=r8079635#L9',
                required=False
            )
            bin_meta_mode = sdk2.parameters.String(
                'Bin meta mode: from here: https://a.yandex-team.ru/arc/trunk/arcadia/ads/quality/adv_machine/lib/runtime/bin_meta/modes/modes.h?rev=8079635&blame=true#L4',
                required=False
            )
            need_decompress_request = sdk2.parameters.Bool('NeedDecompressRequest on meta', required=False)
            timeout_milliseconds = sdk2.parameters.Integer('Response timeout milliseconds', required=False)
            rps = sdk2.parameters.Integer('Shelling RPS', required=False)
            request_retries = sdk2.parameters.Integer('Shelling request retries count', required=False)
            request_count = sdk2.parameters.Integer('Shelling request count', required=False)
            max_shelling_instances_per_dc = sdk2.parameters.Integer('Max shelling instances per DC', required=False)

        with sdk2.parameters.Output():
            services_shelling_result = sdk2.parameters.String('Service shelling result', multiline=True)

        yt = YTParameters

        ext_params = binary_task.binary_release_parameters(stable=True)

    def on_execute(self):
        super(ShellAdvMachineRuntimeExp, self).on_execute()
        configs_proto = self._get_configs_proto()
        self._prepare_ammo(configs_proto)
        self._shell_services(configs_proto)

    def _prepare_ammo(self, configs_proto):
        logger.info('Prepare ammo')
        bin_meta_infos = self._get_bin_meta_infos_from_resource('/bin_meta_infos_config')
        service_id_to_ammo_file = {}
        params = []
        request_count = 0
        for config_proto in configs_proto:
            service_id = self._get_service_id(bin_meta_infos, config_proto.ServiceName)
            if service_id not in service_id_to_ammo_file:
                service_id_to_ammo_file[service_id] = join('.', str(uuid.uuid4()))
                params.extend(['--service', service_id, '-o', service_id_to_ammo_file[service_id]])
            config_proto.ShellingConfig.AmmoFilePath = service_id_to_ammo_file[service_id]
            request_count = max(request_count, config_proto.ShellingConfig.RequestCount)
        am_prepare_worker_shooting_bin = join(str(sdk2.ResourceData(self.Parameters.worker_tools).path), 'am_prepare_worker_shooting')
        env = get_yt_config(self.Parameters.yt, self.author)
        with process_wrapper(self, logger='prepare_ammo') as pl:
            sp.check_call([
                am_prepare_worker_shooting_bin, 'prepare-ammo-for-service',
                '-c', str(request_count),
            ] + params, stdout=pl.stdout, stderr=pl.stderr, env=env)

    def _json_config_from_proto(self, config_proto):
        from ads.quality.adv_machine.lib.priemka.exps.config_loader.proto import config_pb2
        from library.python.protobuf.json import proto2json

        converter_config = proto2json.Proto2JsonConfig(format_output=True, field_name_mode=proto2json.FldNameMode.FieldNameOriginalCase)
        return proto2json.Proto2JsonConverter(config_pb2.TShellingExpConfig, converter_config).convert(config_proto)

    def _shell_services(self, configs_proto):
        logger.info('Shell services')
        am_priemka_bin = join(str(sdk2.ResourceData(self.Parameters.runtime_bins).path), 'am_priemka')
        env = get_yt_config(self.Parameters.yt, self.author)
        successful_shelling_services = []
        failed_shelling_services = []

        for config_proto in configs_proto:
            logger.info('Shell %s', config_proto.ServiceName)
            config_json = self._json_config_from_proto(config_proto)
            logger.info('Config json:\n%s', config_json)
            try:
                with sdk2.helpers.ProcessLog(self, logger='shelling_service_%s' % self._get_py_service_name(config_proto)) as pl:
                    sp.check_call([
                        am_priemka_bin, 'bin_meta_shelling_with_exp_ammo',
                        '--config-json-text', config_json,
                    ], stdout=pl.stdout, stderr=pl.stderr, env=env)
            except Exception as e:
                logger.error('Shelling service %s failed with error:\n%s', config_proto.ServiceName, str(e))
                failed_shelling_services.append(config_proto.ServiceName)
            finally:
                logger.info('Shelling service %s is successful', config_proto.ServiceName)
                successful_shelling_services.append(config_proto.ServiceName)

        self.Parameters.services_shelling_result = 'Shelling services result:\n%s\n\n%s' % (
            '\n'.join(['Shelling service %s OK.' % service_name for service_name in successful_shelling_services]),
            '\n'.join(['Shelling service %s failed. See logs for details.' % service_name for service_name in failed_shelling_services])
        )

        if failed_shelling_services:
            raise common.errors.TaskFailure('Shelling some services failed: %s' % ', '.join(failed_shelling_services))

    def _get_configs_proto(self):
        from ads.quality.adv_machine.lib.priemka.exps.config_loader.proto import config_pb2

        logger.info('Generate config')
        if not self.Parameters.worker_tools:
            raise common.errors.TaskFailure('Not last released AdvMachineWorkerTools resource')
        if not self.Parameters.runtime_bins:
            raise common.errors.TaskFailure('Not last released AdvMachineRuntimeBin resource')

        configs_proto = []
        for service_name in self.Parameters.service_names:
            config_proto = config_pb2.TShellingExpConfig()
            config_proto.ServiceName = service_name
            config_proto.ShellingConfig.ABExpJsonText = self.Parameters.ab_config
            for param_field, proto_field in [
                ('ammo_format', 'AmmoFormat'),
                ('bin_meta_mode', 'BinMetaMode'),
                ('need_decompress_request', 'NeedDecompressRequest'),
                ('timeout_milliseconds', 'TimeoutMilliseconds'),
                ('rps', 'RPS'),
                ('request_retries', 'RetryCount'),
                ('request_count', 'RequestCount'),
                ('max_shelling_instances_per_dc', 'MaxShellingInstancesPerDC'),
            ]:
                value = getattr(self.Parameters, param_field)
                if value:
                    setattr(config_proto.ShellingConfig, proto_field, value)

            configs_proto.append(config_proto)

        return configs_proto

    def _get_py_service_name(self, config_proto):
        return config_proto.ServiceName.replace('-', '_')

    def _get_bin_meta_infos_from_resource(self, resource_key):
        from ads.quality.adv_machine.lib.priemka.exps.config_loader.proto.helper_pb2 import TBinMetaInfos
        from library.python.protobuf.json import json2proto
        from library.python import resource

        resource_config = resource.find(resource_key)
        assert resource_config, 'No %s' % resource_key
        proto_config = TBinMetaInfos()
        json2proto.json2proto(resource_config, proto_config, json2proto.Json2ProtoConfig(field_name_mode=json2proto.FldNameMode.FieldNameOriginalCase))
        return proto_config

    def _get_service_id(self, bin_meta_infos, service_name):
        for infos in bin_meta_infos.BinMetaInfos:
            info = next(iter([service_info for service_info in infos.Services if service_info.ServiceName == service_name]), None)
            if info:
                return info.AmmoServiceID or infos.AmmoServiceID or None
