# -*- coding: utf-8 -*-

import os
from shutil import copy2
import logging

from sandbox import common, sdk2
import sandbox.common.types.resource as ctr

from sandbox.projects.adv_machine.common import (
    AdvMachineDownloadFromYtTask, AdvMachineDownloadFromYtTaskBase
)
from sandbox.projects.adv_machine.common.resources import (
    AdvMachineDataDssm, AdvMachineDataVersionedDssm, AdvMachineDataHitModelsDssm, AdvMachineDssmVersionController
)


logger = logging.getLogger(__name__)


class MakeAdvMachineDssm(AdvMachineDownloadFromYtTask):

    """Base class to create dssm resources"""

    class Requirements(AdvMachineDownloadFromYtTask.Requirements):
        disk_space = 100000

    class Parameters(AdvMachineDownloadFromYtTask.Parameters):
        yt_path = AdvMachineDownloadFromYtTask.Parameters.yt_path(default='adv_machine/dssm')
        dir_name = AdvMachineDownloadFromYtTask.Parameters.dir_name(default='dssm')
        version_controller_path = AdvMachineDownloadFromYtTask.Parameters.version_controller_path(default='dssm_version_controller.json')
        release_services = AdvMachineDownloadFromYtTask.Parameters.release_services(choices=[(s, s) for s in AdvMachineDataDssm.release_services])
        kill_timeout = 12 * 3600

        solomon_project = AdvMachineDownloadFromYtTask.Parameters.solomon_project(default='adv_machine')
        solomon_cluster = AdvMachineDownloadFromYtTask.Parameters.solomon_cluster(default='dssm_learn')
        solomon_service = AdvMachineDownloadFromYtTask.Parameters.solomon_service(default='main_robot')

        old_version_controller_res_id = sdk2.parameters.LastReleasedResource(
            'old version controller resource',
            resource_type=AdvMachineDssmVersionController,
            state=(ctr.State.READY, ),
            required=False,
        )

        create_version_controller = sdk2.parameters.Bool('Create version controller resource', default=True)

    def on_execute(self):
        os.mkdir(self.Parameters.dir_name)
        AdvMachineDataDssm(self, 'AdvMachine dssm: {}'.format(self.Parameters.description), self.Parameters.dir_name)
        if self.Parameters.create_version_controller:
            AdvMachineDssmVersionController(self, 'AdvMachineVersionController dssm: {}'.format(self.Parameters.description), self.Parameters.version_controller_path)
        super(MakeAdvMachineDssm, self).on_execute()

    def _get_st_message(self, resources, release_services):
        return 'Готовы новые dssm. Скоро будут обновлены на {}. \nАвтор кто:{}. \nТаск {}'.format(
            ', '.join(release_services),
            self.author,
            common.utils.get_task_link(self.id),
        )

    def _get_telegram_message(self, params, resources, release_services):
        return '{} чух-чух на {}. Покатил {} потому что {}. Таск {}'.format(
            ', '.join([str(r.type) for r in resources]),
            ', '.join(release_services),
            params['releaser'],
            params['release_comments'] or self.Parameters.description,
            common.utils.get_task_link(self.id),
        )


class MakeAdvMachineUnversionedDssm(MakeAdvMachineDssm):

    """Make dssm resource with unversioned dssms"""

    class Parameters(AdvMachineDownloadFromYtTask.Parameters):
        yt_path = AdvMachineDownloadFromYtTask.Parameters.yt_path(default='adv_machine/dssm')
        dir_name = AdvMachineDownloadFromYtTask.Parameters.dir_name(default='dssm')
        exclude_versioned_resources = AdvMachineDownloadFromYtTask.Parameters.exclude_versioned_resources(default=True)
        exclude_unversioned_resources = AdvMachineDownloadFromYtTask.Parameters.exclude_unversioned_resources(default=False)
        release_services = AdvMachineDownloadFromYtTask.Parameters.release_services(choices=[(s, s) for s in AdvMachineDataDssm.release_services])
        kill_timeout = 12 * 3600
        create_version_controller = False


class MakeAdvMachineVersionedDssm(MakeAdvMachineDssm):

    """Make dssm resource with versioned dssms"""

    class Parameters(MakeAdvMachineDssm.Parameters):
        exclude_unversioned_resources = MakeAdvMachineDssm.Parameters.exclude_unversioned_resources(default=True)
        exclude_versioned_resources = MakeAdvMachineDssm.Parameters.exclude_versioned_resources(default=False)
        create_version_controller = sdk2.parameters.Bool('Create version controller resource', default=False)

    def on_execute(self):
        os.mkdir(self.Parameters.dir_name)
        AdvMachineDataVersionedDssm(self, 'AdvMachine dssm: {}'.format(self.Parameters.description), self.Parameters.dir_name)
        if self.Parameters.create_version_controller:
            AdvMachineDssmVersionController(self, 'AdvMachineVersionController dssm: {}'.format(self.Parameters.description), self.Parameters.version_controller_path)
        super(MakeAdvMachineDssm, self).on_execute()


class MakeAdvMachineHitModelsDssm(AdvMachineDownloadFromYtTaskBase):

    """Make hit models dssm resources with dssms"""

    class Parameters(AdvMachineDownloadFromYtTaskBase.Parameters):
        yt_path = AdvMachineDownloadFromYtTaskBase.Parameters.yt_path(default='adv_machine/dssm')
        dir_name = AdvMachineDownloadFromYtTaskBase.Parameters.dir_name(default='dssm')
        strip_version = sdk2.parameters.Bool('Strip version for versioned models', default=True)
        files_whitelist = sdk2.parameters.List('Models to download (empty means no restrictions)', required=True, default=['spyhit_sort_2.apply', 'spyhit_rt.apply', 'spy_hit_context.apply',
                                                                                                                           'spyhit_rmp_rt.apply', 'spyhit_rt_rmp_context.apply'])
        bundled_models = sdk2.parameters.Dict('Models lists to store as single resource', required=False, default={
            'spy_hit_context': 'spyhit_rt.apply,spy_hit_context.apply',
        })

    def get_command_parameters(self):
        cmd = [
            '--separate-dirs'
        ]
        if self.Parameters.strip_version:
            cmd.append('--strip-version')
        if self.Parameters.files_whitelist:
            cmd.append('--files-whitelist')
            cmd.extend(self.Parameters.files_whitelist)
        return cmd

    def on_execute(self):
        os.mkdir(self.Parameters.dir_name)
        model_to_file_dir = dict()
        for model in self.Parameters.files_whitelist:
            model_dir = os.path.join(self.Parameters.dir_name, model)
            model_to_file_dir[model] = model_dir
            AdvMachineDataHitModelsDssm(self, 'AdvMachine dssm: {}'.format(self.Parameters.description), model_dir, model=model)

        bundled_model_to_files_dir = dict()
        for bundle_name in self.Parameters.bundled_models:
            bundle_files_dir = os.path.join(self.Parameters.dir_name, bundle_name)
            bundled_model_to_files_dir[bundle_name] = bundle_files_dir
            AdvMachineDataHitModelsDssm(self, 'AdvMachine dssm bundle: {}'.format(self.Parameters.description), bundle_files_dir, model=bundle_name)

        super(MakeAdvMachineHitModelsDssm, self).on_execute()

        for bundle_name, models in self.Parameters.bundled_models.items():
            bundle_files_dir = bundled_model_to_files_dir[bundle_name]
            os.mkdir(bundle_files_dir)
            for model in models.split(','):
                model_dir = model_to_file_dir[model]
                for model_dir_content in os.listdir(model_dir):
                    copy2(os.path.join(model_dir, model_dir_content), bundle_files_dir)
