# -*- coding: utf-8 -*-

import logging
import time

from sandbox import sdk2
from sandbox import common
import sandbox.common.types.resource as ctr

from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.adv_machine.common import process_wrapper
from sandbox.projects.adv_machine.common import AdvMachineReleaseTask
from sandbox.projects.adv_machine.common.resources import AdvMachineDataSelectTypeJson, adv_machine_releasers, adv_machine_subscribers


logger = logging.getLogger(__name__)


class AdvMachineRsyaExpToolsBin(sdk2.Resource):
    releasable = True
    any_arch = True
    auto_backup = True
    releasers = adv_machine_releasers
    release_subscribers = adv_machine_subscribers
    __default_attribute__ = sdk2.parameters.String


class MakeAdvMachineSelectTypeJson(AdvMachineReleaseTask):

    """Create adv machine select_type.json"""

    class Requirements(AdvMachineReleaseTask.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(AdvMachineReleaseTask.Parameters):
        with sdk2.parameters.Group('YT parameters') as yt_block:
            yt_proxy = sdk2.parameters.String('YT proxy', required=True, default='hahn')
            yt_token_vault = sdk2.parameters.String('YT_TOKEN vault name', default='adv_machine_yt_token')
            select_type_table_path = sdk2.parameters.String('yabsdb/SelectType table', required=True, default='//home/yabs/dict/SelectType')

        am_rsya_exp_tools_res_id = sdk2.parameters.LastReleasedResource(
            'am_rsya_exp_tools binary resource',
            resource_type=AdvMachineRsyaExpToolsBin,
            state=(ctr.State.READY, ),
            required=True,
        )

        timestamp_to_wait = sdk2.parameters.Integer('unix timestamp to wait replica, use 0 to do task immediately, default value is now', default=None)

        release_services = AdvMachineReleaseTask.Parameters.release_services(choices=[(s, s) for s in AdvMachineDataSelectTypeJson.release_services])

        select_type = sdk2.parameters.ParentResource('Select type resource', required=False)

        skip_checks = sdk2.parameters.Bool('Don\'t check file', default=True)

    @staticmethod
    def convert_timestamp_to_wait(timestamp_param):
        if timestamp_param is None:
            return int(time.time())
        else:
            return timestamp_param

    def on_execute(self):
        bin_res = sdk2.ResourceData(self.Parameters.am_rsya_exp_tools_res_id)

        select_type_json_path = 'select_type.json'
        if self.Parameters.select_type:
            select_type_json_path = str(self.Parameters.select_type.path)
        else:
            AdvMachineDataSelectTypeJson(self, 'AdvMachine select_type.json: {}'.format(self.Parameters.description), select_type_json_path)

        cmd = [
            str(bin_res.path), 'download-experiment-file-from-MR',
            '-s', self.Parameters.yt_proxy,
            '-i', self.Parameters.select_type_table_path,
            '-o', select_type_json_path,
            '-t', str(self.convert_timestamp_to_wait(self.Parameters.timestamp_to_wait)),
        ]
        if self.Parameters.skip_checks:
            cmd += ['--force-pass-checks']

        env = {'MR_RUNTIME': 'YT'}
        if self.Parameters.yt_token_vault:
            env['YT_TOKEN'] = sdk2.Vault.data(self.Parameters.yt_token_vault)

        with process_wrapper(self, logger='am_rsya_exp_tools') as pl:
            try:
                sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr, env=env)
            finally:
                with pl.stdout.path.open() as f:
                    self.set_info(f.read())

        super(MakeAdvMachineSelectTypeJson, self).on_execute()

    def _get_st_message(self, resources, release_services):
        return 'Готов новый selectType.json. Скоро будет обновлен на {}. \nАвтор кто:{}. \nТаск {}'.format(
            ', '.join(release_services),
            self.author,
            common.utils.get_task_link(self.id),
        )

    def _get_telegram_message(self, params, resources, release_services):
        return 'SelectType.json чух-чух на {}. Покатил {} потому что {}. Таск {}'.format(
            ', '.join(release_services),
            params['releaser'],
            params['release_comments'] or self.Parameters.description,
            common.utils.get_task_link(self.id),
        )
