# -*- coding: utf-8 -*-

import os
import logging

from os.path import join as pj

import sandbox.common.types.resource as ctr
from sandbox import common, sdk2

from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.adv_machine.common import process_wrapper
from sandbox.projects.adv_machine.common import AdvMachineReleaseTask
from sandbox.projects.adv_machine.common.parameters import YTParameters
from sandbox.projects.adv_machine.common.resources import TurboappDicts, TurboappOfferInfos, SuperappMakeDictsBin
from sandbox.projects.adv_machine.common.yt_params import get_yt_config


logger = logging.getLogger(__name__)


class MakeTurboappDicts(AdvMachineReleaseTask):

    """Create turboapp dicts resources"""

    class Requirements(AdvMachineReleaseTask.Requirements):
        ram = 200 * 1024
        disk_space = 120 * 1024

    class Parameters(AdvMachineReleaseTask.Parameters):
        yt = YTParameters
        turboapps_table = sdk2.parameters.String('turboapps table')
        turbo_table = sdk2.parameters.String('turbo table')
        make_dicts_res_id = sdk2.parameters.LastReleasedResource(
            'superapp_make_dicts binary resource',
            resource_type=SuperappMakeDictsBin,
            state=(ctr.State.READY, ),
            required=True,
        )

    def on_execute(self):
        os.mkdir('turboapp_dicts')
        os.mkdir('id2offers')

        bin_res = sdk2.ResourceData(self.Parameters.make_dicts_res_id)

        cmd = [
            pj(str(bin_res.path), 'superapp_make_dicts'), 'build-dicts',
            '--dst-dict-file', pj('turboapp_dicts', 'apps.bin'),
            '--dst-baseurl-trie-file', pj('turboapp_dicts', 'baseurl2app.bin'),
            '--dst-turbo-trie-file', pj('turboapp_dicts', 'url2app.bin'),
            '--dst-offers-trie-file', pj('turboapp_dicts', 'url2offer.bin'),
            '--dst-offer-infos-file', pj('id2offers', 'id2offers.trie'),
            '--server', self.Parameters.yt.proxy,
            '--vvv'
        ]

        if self.Parameters.turboapps_table:
            cmd.extend(['-a', self.Parameters.turboapps_table])

        if self.Parameters.turbo_table:
            cmd.extend(['-t', self.Parameters.turbo_table])

        env = get_yt_config(self.Parameters.yt, self.author)

        with process_wrapper(self, logger='superapp_make_dicts') as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr, env=env)

        TurboappDicts(self, 'AdvMachine turboapp dicts: {}'.format(self.Parameters.description), 'turboapp_dicts')
        TurboappOfferInfos(self, 'AdvMachine offer infos: {}'.format(self.Parameters.description), pj('id2offers', 'id2offers.trie'))

        super(MakeTurboappDicts, self).on_execute()
