# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
import sandbox.common.types.resource as ctr

from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.adv_machine.common import process_wrapper, get_yt_config
from sandbox.projects.adv_machine.common.resources import adv_machine_subscribers, adv_machine_releasers
from sandbox.projects.adv_machine.common.parameters import YTParameters


logger = logging.getLogger(__name__)


class SolomonMonitoringYtQuotaBin(sdk2.Resource):
    releasable = True
    any_arch = True
    auto_backup = True
    releasers = adv_machine_releasers
    release_subscribers = adv_machine_subscribers
    __default_attribute__ = sdk2.parameters.String


class SolomonMonitoringYtQuota(sdk2.Task):

    """Regular task to monitoring yt quota for advquality"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        yt = YTParameters

        with sdk2.parameters.Group('Monitoring parameters') as monitoring_block:
            yt_roots = sdk2.parameters.List('YT roots')
            yt_quota = sdk2.parameters.String('YT quota', required=True)

            solomon_project = sdk2.parameters.String('Solomon project')
            solomon_cluster = sdk2.parameters.String('Solomon cluster')
            solomon_service = sdk2.parameters.String('Solomon service')

            monitoring_resource_id = sdk2.parameters.LastReleasedResource(
                'Monitoring YT quota binary resource',
                resource_type=SolomonMonitoringYtQuotaBin,
                state=(ctr.State.READY, ),
                required=True,
            )

    def on_execute(self):
        bin_res = sdk2.ResourceData(self.Parameters.monitoring_resource_id)

        cmd = [
            str(bin_res.path),
            '-q', self.Parameters.yt_quota,
            '--yt-proxy', self.Parameters.yt.proxy,
            '--yt-prefix', self.Parameters.yt.prefix,
        ]

        if self.Parameters.yt_roots:
            cmd.extend(['-r', ' -r '.join(self.Parameters.yt_roots)])

        if self.Parameters.solomon_cluster:
            cmd.extend(['--cluster', self.Parameters.solomon_cluster])

        if self.Parameters.solomon_project:
            cmd.extend(['--project', self.Parameters.solomon_project])

        if self.Parameters.solomon_service:
            cmd.extend(['--service', self.Parameters.solomon_service])

        env = get_yt_config(self.Parameters.yt, self.author)

        secret = sdk2.yav.Secret('sec-01dq7yfgbkh83tx9jk2rsgyggw')
        solomon_token = secret.data()['token']
        env.update({'SOLOMON_TOKEN': solomon_token})

        with process_wrapper(self, logger='monitoring') as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr, env=env)
