# -*- coding: utf-8 -*-
import logging
import shutil

from os.path import join
from sandbox import sdk2
from sandbox import common

import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

from sandbox.projects.adv_machine.common import get_yt_config, process_wrapper, resources, release_task
from sandbox.projects.adv_machine.common.parameters import YTParameters
from sandbox.projects.common import binary_task
from sandbox.sdk2.helpers import subprocess as sp


logger = logging.getLogger(__name__)


class PrepareAdvMachineAmmo(release_task.AdvMachineReleaseTaskServiceName, binary_task.LastBinaryTaskRelease, sdk2.Task):

    """Задача для сборки патронов для сервисов садовой машины"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        disk_space = 10 * 1024

    class Parameters(release_task.AdvMachineReleaseTaskServiceName.Parameters, sdk2.Parameters):

        with sdk2.parameters.Group('Input resources') as input_res_block:
            worker_tools = sdk2.parameters.LastReleasedResource(
                'Ammo prepare bin',
                resource_type=resources.AdvMachineWorkerTools,
                state=(ctr.State.READY, ),
                required=True,
            )

        with sdk2.parameters.Group('Optional parameters') as optional_param_block:
            request_count = sdk2.parameters.Integer('Shelling request count', required=True)

        yt = YTParameters

        ext_params = binary_task.binary_release_parameters(stable=True)

    def on_execute(self):
        super(PrepareAdvMachineAmmo, self).on_execute()
        self._prepare_ammo()

    def on_release(self, parameters):
        super(PrepareAdvMachineAmmo, self).on_release(parameters)
        service_names = self.get_services()
        if len(service_names) > 1:
            raise common.errors.TaskFailure('service_names > 1')
        self.mark_released_resources(parameters["release_status"], ttl=15)

    def _prepare_ammo(self):
        logger.info('Prepare ammo')

        params = []

        service_names = self.get_services()

        for service_name in service_names:
            params.extend(['--service', service_name, '-o', service_name + "-ammo" + '.o'])

        am_prepare_worker_shooting_bin = join(str(sdk2.ResourceData(self.Parameters.worker_tools).path), 'am_prepare_worker_shooting')
        env = get_yt_config(self.Parameters.yt, self.author)
        with process_wrapper(self, logger='prepare_ammo') as pl:
            sp.check_call([
                am_prepare_worker_shooting_bin, 'prepare-ammo-for-service',
                '-c', str(self.Parameters.request_count),
            ] + params, stdout=pl.stdout, stderr=pl.stderr, env=env)

        for service_name in service_names:
            resource = resources.AdvMachineAmmo(self, "ammo for " + service_name, service_name + ".o")
            resource.service_name = service_name
            shutil.copyfile(service_name + "-ammo.o", service_name + ".o")
