# -*- coding: utf-8 -*-
import os
import re

from sandbox import sdk2
from sandbox import common
import sandbox.common.types.task as ctt

from sandbox.projects.adv_machine.common.resources import AdvMachineFormulas
from sandbox.projects.adv_machine.common.release_task import AdvMachineReleaseTask

from sandbox.projects.adv_machine.make_formulas import MakeAdvMachineFormulas
from sandbox.projects.adv_machine.make_select_type_json import MakeAdvMachineSelectTypeJson, AdvMachineDataSelectTypeJson


class ReleaseAdvMachineSelectTypeWithFormulas(AdvMachineReleaseTask):

    """Задача для подготовки, сборки, проверки индекса AdvMachine"""

    class Requirements(AdvMachineReleaseTask.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(AdvMachineReleaseTask.Parameters):
        release_services = AdvMachineReleaseTask.Parameters.release_services(choices=[(s, s) for s in AdvMachineFormulas.release_services])
        timestamp_to_wait = MakeAdvMachineSelectTypeJson.Parameters.timestamp_to_wait()

    def on_execute(self):
        subtasks = list(self.find().limit(0))
        with self.memoize_stage.first:
            select_type = AdvMachineDataSelectTypeJson(self, self.Parameters.description, 'select_type.json')
            self.Context.select_type_id = select_type.id
            formulas = AdvMachineFormulas(self, self.Parameters.description, 'formulas')
            self.Context.formulas_id = formulas.id

            if not subtasks:
                formulas_task = MakeAdvMachineFormulas(
                    self,
                    owner=self.Parameters.owner,
                    priority=self.Parameters.priority,
                    description=self.Parameters.description,
                    deploy_after_release=True,
                    release_services=self.Parameters.release_services,
                    additional_release_services=self.Parameters.additional_release_services,
                    timestamp_to_wait=self.Parameters.timestamp_to_wait,
                    formulas=formulas,
                    dir_name=formulas.path
                ).enqueue()

                select_type_task = MakeAdvMachineSelectTypeJson(
                    self,
                    owner=self.Parameters.owner,
                    priority=self.Parameters.priority,
                    description=self.Parameters.description,
                    deploy_after_release=True,
                    release_services=self.Parameters.release_services,
                    additional_release_services=self.Parameters.additional_release_services,
                    select_type=select_type
                ).enqueue()

                raise sdk2.WaitTask([formulas_task, select_type_task], list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK), True)

        failed_subtasks = [t for t in subtasks if t.status in ctt.Status.Group.SCHEDULER_FAILURE]
        if failed_subtasks:
            raise common.errors.TaskError(
                '\n'.join(['Subtask {} ({}) was finished with the status of {}'.format(t.type, t.id, t.status) for t in failed_subtasks])
            )

        select_type = sdk2.ResourceData(sdk2.Resource.find(AdvMachineDataSelectTypeJson, id=self.Context.select_type_id).first())
        formulas = sdk2.ResourceData(sdk2.Resource.find(AdvMachineFormulas, id=self.Context.formulas_id).first())

        formula_names = self._get_formula_names(str(select_type.path))
        self.set_info('Formulas:\n{}'.format('\n'.join(formula_names)))

        no_exists_formulas = self._check_formulas(str(formulas.path), formula_names)

        if no_exists_formulas:
            raise common.errors.TaskFailure('No exists formulas:\n {}'.format('\n'.join(no_exists_formulas)))

    @staticmethod
    def _get_formula_names(select_type_path):
        with open(select_type_path, 'rb') as stream:
            select_type = stream.read()

        names = set()
        for m in re.finditer('\\\\"Formulas\\\\":', select_type):
            start = select_type.find('\"', m.end()) + 1
            end = select_type.find('\"', start)
            names.update(select_type[start:end].rstrip('\\').split(','))

        return names

    @staticmethod
    def _check_formulas(formulas_path, formula_names):
        return {name for name in formula_names if not name.startswith('sf_') and not os.path.exists(os.path.join(formulas_path, '{}.amf'.format(name)))}
