# -*- coding: utf-8 -*-
import cgi
import json
import logging
from contextlib import closing
from urllib2 import urlopen, HTTPError
from urlparse import urljoin

import time

import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.projects.advq.artifacts import ADVQ_DATATEST_TOOL
from sandbox.projects.advq.common.parameters import AdvqBuildBinariesResourceParameter
from sandbox.projects.advq.common import AdvqApiDataTestReport


class AdvqApiDataTest(sdk2.Task):
    """
    Проверяет на фиксированном списке запросов, что ответы от двух разных API не сильно отличаются.

    Успех/неуспех определяется выставленным выходным параметром result_success, а ID ресурса
    с HTML-отчётом в result_report_res_id.
    """

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.LINUX_TRUSTY & ctc.Tag.IPV6
        cores = 1
        disk_space = 2 * 1024
        ram = 128

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        advq_build_binaries = AdvqBuildBinariesResourceParameter('ADVQ generation utils resource', required=True)
        test_config = sdk2.parameters.String('Configuration for testing', required=True)
        release_url = sdk2.parameters.Url('URL of released API', required=True,
                                          default='http://back.advq.yandex.ru/advq')
        testing_url = sdk2.parameters.Url('URL of testing API', required=True,
                                          default='http://back.advq.yandex.ru/advq_test')
        minimal_check = sdk2.parameters.Bool("Perform only minimal check", default=False)
        # защита от проблем с DNS и прочих спамов и куков -- пытаемся несколько раз перезапустить бинарь
        retry_count = sdk2.parameters.Integer("Retry count", default=5, required=True)
        with sdk2.parameters.Output:
            result_success = sdk2.parameters.Bool("True on success")
            result_report_res_id = sdk2.parameters.Integer("HTML report resource ID")

    def on_execute(self):
        result_html = 'result.html'
        try:
            released_meta = self._get_meta(str(self.Parameters.release_url))
            testing_meta = self._get_meta(str(self.Parameters.testing_url))
        except HTTPError as ex:
            with open(result_html, 'wb') as report_out:
                report_out.write("""
                <p>Request for info {url} failed ({http_code}):</p>
                <pre>
                {body}
                </pre>
                """.format(
                    url=cgi.escape(ex.url),
                    http_code=str(ex.code),
                    body=cgi.escape(ex.read()),
                ))
            result_resource = AdvqApiDataTestReport(
                self,
                'Data test report {} => {}'.format(self.Parameters.release_url, self.Parameters.testing_url),
                result_html,
            )
            result_resource.released_meta = 'NONE'
            result_resource.testing_meta = 'NONE'
            result_resource.test_success = False
            result_resource.tested_config = self.Parameters.test_config
            sdk2.ResourceData(result_resource).ready()

            self.Parameters.result_success = False
            self.Parameters.result_report_res_id = result_resource.id

            return

        if not self.Parameters.minimal_check:
            binaries = sdk2.ResourceData(self.Parameters.advq_build_binaries)
            args = [
                str(binaries.path.joinpath(ADVQ_DATATEST_TOOL)),
                '--output-html={}'.format(result_html),
                '--release-url={!s}'.format(self.Parameters.release_url),
                '--testing-url={!s}'.format(self.Parameters.testing_url),
                '--timeout={}'.format(int(self.Parameters.kill_timeout) - 10),
            ]

            success = True

            retry_count = self.Parameters.retry_count
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(__name__)) as pl:
                while retry_count > 0:
                    try:
                        sdk2.helpers.subprocess.check_call(args, stdout=pl.stdout, stderr=pl.stderr)
                        break
                    except sdk2.helpers.subprocess.CalledProcessError as err:
                        if err.returncode == 100:  # Этот код возвращается при непрошедших тестах
                            success = False
                            break
                        else:
                            retry_count -= 1
                            if retry_count == 0:
                                raise  # попытки исчерпаны, пробрасываем исключение наружу
                            else:
                                time.sleep(60)

            result_resource = AdvqApiDataTestReport(
                self,
                'Data test report {} => {}'.format(released_meta, testing_meta),
                result_html,
            )
            result_resource.released_meta = released_meta
            result_resource.testing_meta = testing_meta
            # в атрибуты отчета положить статус прогона и конфиг
            result_resource.test_success = success
            result_resource.tested_config = self.Parameters.test_config
            sdk2.ResourceData(result_resource).ready()

            self.Parameters.result_success = success
            self.Parameters.result_report_res_id = result_resource.id
        else:
            self.Parameters.result_success = True
            self.Parameters.result_report_res_id = None

    def _get_meta(self, url):
        info_url = urljoin(url.rstrip('/') + '/', 'info') + '?format=json'
        with closing(urlopen(info_url)) as stream:
            return json.load(stream)['service'].strip()
